package netobj;

import java.io.*;
import java.net.*;

// The class ServerConnection represents a connection to a server JVM.
// This connection is used to request remote invocations of objects
// owned by the server JVM and issued by the local JVM.
// The objects of ServerConnection are created by the manager the
// first time that a reference arrives from a remote JVM.
// To avoid blocking the manager, the socket required by the connection
// is opened in the background by a short lived thread.
class ServerConnection implements Runnable {
  private static Manager manager= Manager.getLocalManager();
  final static long TAG= 0x49514a4b91f91ae3L; // just for checking the socket

  // The identification of the server JVM 
  private Space sp;

  // The state of the connection
  private Exception excp= null;  // the exception that broke the socket
  private Thread owner= null;
          // when this field is null, the connection is free.
          // When it is not null, the thread referenced has
          // taken the connection and therefore any other
          // thread willing to use the connection has to wait.

  // The streams for communication
  private Socket socket= null;   // a null value means that it must be open
  private ObjectOutputStream out;
  private ObjectInputStream in; 

  // Constructor: server connections are created by the manager
  // when an object is received from a JVM for which there is no
  // connection.
  ServerConnection(Space sp) {
    this.sp= sp;
    Debug.prt("ServerConnection: Creating a connection with server "+
                       sp);

    // The connection is established by a short-lived thread in the
    // background to avoid blocking the manager.
  }

  // The body of the thread which establishes the connection.
  // This thread is launched by the method take when there is no connection
  // established (for example when a remote invocation breaks the socket).
  public synchronized void run() {
    Socket socket;
    try {
      try {
        // Opens the socket
        socket= new Socket(sp.iaddr, sp.port);         // throws IOException
        out= new ObjectOutputStream(socket.getOutputStream()); // throws id.
        in= new ObjectInputStream(socket.getInputStream());    // throws id.
        // Sends the local space identification through the socket
        out.writeObject(new ConnectionFactoryImpl());
      }
      catch (IOException excp) {
        System.err.println("ServerConnection: the socket could not be open");
        excp.printStackTrace();
        this.excp= excp;
        return;
      }

      this.socket= socket; // success!
    }
    finally {
      release(); // This thread starts with the connection taken,
                 // so at the end we must release it
    }
    // The thread finishes
  }

  static class ConnectionFactoryImpl implements ConnectionFactory {
    public void makeConnection(Socket socket,
                   ObjectInputStream in, ObjectOutputStream out) {
      new ClientConnection(socket, in, out);
    }
  }
  
  // take: take this connection for exclusive use of the calling thread.
  // It also checks if there is a working socket for this connection.
  // If the socket is not working it tries to open it.
  synchronized void take() throws NetException {
    try {
      // the connection might be taken by a thread opening the
      // socket or by a thread perfoming a remote call.
      while (owner!=null) // wait until the connection be released
        wait();
      // if the socket is operational, use it.
      if (socket==null) {
        // if there is no open socket and nobody has taken the connection
        // to open it, create a new thread to open the socket.
        Debug.prt("ServerConnection: launching thread for "+
                           "establishing connection with "+sp);
        owner= new Thread(this);
        owner.start();
        while (owner!=null) // wait until the socket be released
          wait();
        // No more oportunities if the socket could not be opened.
        // (but if another thread took the connection and broke it
        // ... bad luck).
      }
    }
    catch (InterruptedException excp) {
      System.err.println("ServerConnection: the connection to "+sp+
                         " was interrupted");
    }
    if (socket==null)
      throw new NetException(this.excp);
    owner= Thread.currentThread();
  }
  
  // release: releases the connection so other threads be able to
  // share the same connection.
  synchronized void release() {
    Debug.prt("ServerConnection: connection with server "+sp+" released");
    owner= null;
    notifyAll();
  }

  // startRemoteCall: takes the connection for executing a remote call,
  // so any other thread invoking later this method will wait until the owning
  // thread executes waitRemoteRet<SomeType>.  It also sends the prolog for
  // the remote invocation and returns an object output stream so that the
  // caller sends the parameters.
  public ObjectOutputStream startRemoteCall(WireRep wrep, int methodId)
                            throws NetException {
    take();
    try {
      out.writeObject(wrep);   // throws IOException
      out.writeInt(methodId);  // throws id.
    }
    catch (IOException excp) {
      System.err.println("ServerConnection: I/O error while communicating "+
                         "with host "+sp);
      throwNetException(excp);
    }
    return out;
  }

  // flushAndWait: flushes the output socket and waits for a boolean
  // value on the input socket, indicating if the remote call was
  // successfull or not.  The remote call is successfull when it does not
  // produce any exceptions.
  ObjectInputStream flushAndWait(WireRep wrep) throws NetException {
    try {
      out.writeLong(TAG);       // throws IOException
      out.reset();              // Clean the hash table kept by out
      out.flush();              // throws id.
      Debug.prt("ServerConnection: remote call issued for "+wrep);
      // The following call checks the consistency of the communication
      // protocol (it can be removed)
      if (in.readLong()!=TAG) { // throws id.
        System.err.println("ServerConnection: did not get the expected tag "+
                           "from "+sp);
        throw new IOException("An inconsistency has ocurred while reading "+
                              "from "+sp);
      }
      // read the boolean value indicating the success or failure of the
      // remote call.
      if (!in.readBoolean()) { // throws id.
        Exception excp= (Exception)in.readObject();
        release();
        throw new InvocationException(excp);
      }
    }
    catch (ClassCastException excp) {
      System.err.println("ServerConnection: did not read the expected "+
                         "exception from host "+sp);
      throwNetException(excp);
    }
    catch (ClassNotFoundException excp) {
      System.err.println("ServerConnection: class not found while "+
                         "deserializing data from host "+sp);
      throwNetException(excp);
    }
    catch (IOException excp) {
      System.err.println("ServerConnection: I/O error while communicating "+
                         "with host "+sp);
      throwNetException(excp);
    }
    return in;
  }

  // throwNetException: closes the socket because an error was found
  // and there is no way of knowing if the socket can be recovered.
  // Anyway, on a future remote call, the socket will be reopened.
  public void throwNetException(Exception excp) throws NetException {
    try { socket.close(); }
    catch (IOException excp2) { }
    socket= null;  // Close the socket and
    release();     // Release the connection
    throw new NetException(excp);
  }

  // waitRemoteRetVoid: the stubs will invoke this method after
  // sending the parameters through the socket.  It flushes the socket
  // and waits for the completion of the remote call.  Finally it
  // releases the connection so others threads be able to perform
  // remote calls.
  public void waitRemoteRetVoid(WireRep wrep) throws NetException {
    flushAndWait(wrep);  // throws NetException
    release();       // throws id.
  }

  // waitRemoteInt: like waitRemoteRetVoid, but used to wait for remote
  // methods that return an int.  It gets the value from the input socket.
  public int waitRemoteRetInt(WireRep wrep) throws NetException {
    int retInt= 0;
    flushAndWait(wrep);         // throws NetException
    try {
      retInt= in.readInt(); // throws IOException
    }
    catch (IOException excp) {
      System.err.println("ServerConnection: I/O error while communicating "+
                         "with host "+sp);
      throwNetException(excp);
    }
    release();
    return retInt;
  }

  // waitRemoteDouble: like waitRemoteRetVoid, but used to wait for remote
  // methods that return a double.  It gets the value from the input socket.
  public double waitRemoteRetDouble(WireRep wrep) throws NetException {
    double retDouble= 0;
    flushAndWait(wrep);               // throws NetException
    try {
      retDouble= in.readDouble(); // throws IOException
    }
    catch (IOException excp) {
      System.err.println("ServerConnection: I/O error while communicating "+
                         "with host "+sp);
      throwNetException(excp);
    }
    release();
    return retDouble;
  }

  // waitRemoteBoolean: like waitRemoteRetVoid, but used to wait for remote
  // methods that return a boolean.  It gets the value from the input socket.
  public boolean waitRemoteRetBoolean(WireRep wrep) throws NetException {
    boolean retBoolean= false;
    flushAndWait(wrep);                 // throws NetException
    try {
      retBoolean= in.readBoolean(); // throws IOException
    }
    catch (IOException excp) {
      System.err.println("ServerConnection: I/O error while communicating "+
                         "with host "+wrep.sp);
      throwNetException(excp);
    }
    release();
    return retBoolean;
  }

  // waitRemoteObject: like waitRemoteRetVoid, but used to wait for remote
  // methods that return an object.  It gets the object from the input socket.
  public Object waitRemoteRetObject(WireRep wrep) throws NetException {
    Object retObj= null;
    flushAndWait(wrep);     // throws NetException
    try {
      retObj= in.readObject(); // throws IOException, ClassNotFound...
    }
    catch (ClassNotFoundException excp) {
      System.err.println("ServerConnection: I/O error while communicating "+
                         "with host "+sp);
      throwNetException(excp);
    }
    catch (IOException excp) {
      System.err.println("ServerConnection: serialization error while "+
                         "communicating with host "+sp);
      throwNetException(excp);
    }
    release();
    return retObj;
  }
}
