package netobj;

import java.io.*;
import java.net.*;
import java.util.*;
import java.lang.ref.*;

// The class Manager has only one instance.  It is used for centralizing
// the management of all network objects.
public class Manager {
  // The manager for this JVM (there is only one for each JVM)
  static Manager manager= new Manager();
  static Manager getLocalManager() {
    if (manager==null)
      throw new Error("Manager: non recoverable bug, manager is null!");
    return manager;
  }

  DistGC distGc= null; // The distributed GC implementation

  // Actually, all the variables are private
  Map objTbl= new HashMap();   // The object table
  Set ownSet= new HashSet();   // The set of network objects owned by
                               // this JVM and having remote references
  Map serverConnectionTbl= new HashMap();
                                 // Maps spaces to connections to thats spaces

  Object dummyObj= new Object(); // see getInObjTbl

  int   socketPort= 0; // Which port use for accepting connections.
  Space selfSp= null;  // The local JVM space identifiers
  // selfSpace: returns the identifier of the local JVM space
  Space selfSpace() {
    // selfSp is null when no local net object has been sent remotely.
    // In such case, it would be a bug to invoke selfSpace.
    if (selfSp==null)
      throw new Error("Manager: non recoverable bug, selfSp is null!");
    return selfSp; // it is initialized by the constructor
  }

  //***************************************************************
  // Methods for manipulating the object table

  // getInObjTbl: Tries to get a network object from the object table given
  // its wire representation.  It can return (i) the concrete
  // object if the object is local, (ii) the stub if the object was
  // received previously, or (iii) null if it is the first time the object
  // is received.
  // In case (iii) the dummy object is put in the object table in association
  // with wrep.  The caller will create a stub for the net object.
  // Meanwhile, if other calls to getInObjTbl get the dummy objects from
  // the object table, that calls are put to wait until the stub be created.
  private synchronized NetObjBase getInObjTbl(WireRep wrep) {
    Object ref;
    for(;;) {
      ref= objTbl.get(wrep);
      if (ref!=dummyObj)
        break;
      // netObj is the dummy net object, a dirty call is in progress
      // so we just wait it to conclude
      try { wait(); }
      catch (InterruptedException excp) {
        System.err.println("Manager: interrupted exception");
      }
    }
    if (ref!=null) {
      Object netObj= ((WeakReference)ref).get();
      if (netObj!=null)
        return (NetObjBase)netObj;
    }
    // if the object is not found the dummy object is put in the object
    // table to prevent other threads to start a dirty call
    objTbl.put(wrep, dummyObj);

    return null;
  }

  // getLocalInObjTbl: like getInObjTbl, but it looks for local object.
  ConcreteNetObj getLocalInObjTbl(long id) {
    return (ConcreteNetObj)getInObjTbl(new WireRep(selfSp, id, null));
  }

  // putInObjTbl: put netObj in the object table and notifies
  // any thread waiting for it.
  synchronized void putInObjTbl(NetObjBase netObj) {
    if (netObj instanceof ConcreteNetObj)
      ownSet.add(netObj); // put in the ownSet to avoid the object be GCed.
    objTbl.put(netObj.wrep, new WeakReference(netObj));
    notifyAll(); // Because some threads may be waiting for this object
  }

  // cleanObjTbl: invoked when finalizing a stub, because some time before
  // the local GC detected that the stub was no more reachable, and
  // therefore, the weak reference was cleared.
  // Meantime, some thread may have got a reference of the same object
  // and placed the dummy object or a new weak reference with a new stub
  // for the same object.  In such case, cleanObjTbl does nothing and
  // returns false.  Otherwise, it removes the stub from the object
  // table and returns true.
  synchronized boolean cleanObjTbl(ConcreteNetObjStub stub) {
    Object ref= objTbl.get(stub.wrep);
    if (ref==null || ref==dummyObj)
      return false;
    if (((WeakReference)ref).get()!=null)
      return false;
    objTbl.remove(stub.wrep);
    return true;
  }

  // removeFromObjTbl: deletes a concrete object from the object table.
  synchronized void removeFromObjTbl(ConcreteNetObj netObj) {
    objTbl.remove(netObj.wrep);
    ownSet.remove(netObj);
  }

  // traceObjTbl: shows some statistics about the object table.  It servers
  // for debugging purposes, for example to check that the table
  // does not grow forever.
  synchronized void traceObjTbl() {
    Iterator iter= objTbl.keySet().iterator();
    int tot= 0;
    int expired= 0;
    int dummys= 0;
    while (iter.hasNext()) {
      WireRep wrep= (WireRep)iter.next();
      tot++;
      Object ref= objTbl.get(wrep);
      if (ref==dummyObj)
        dummys++;
      else if (((WeakReference)ref).get()==null)
        expired++;
    }
    Debug.prt("Manager: size of object table="+tot+" expired="+
              expired+" dummys="+dummys+" size of own="+ownSet.size());
  }

  //***************************************************************
  // Methods for receving and sending network references

  // getServerConnection: gets the server connection for the JVM represented
  // by sp if it exists, or establishes the connection if it doesn't exist.
  synchronized ServerConnection getServerConnection(Space sp) {
    ServerConnection server=
                     (ServerConnection)serverConnectionTbl.get(sp);
    if (server==null) {
      server= new ServerConnection(sp);
      serverConnectionTbl.put(sp, server);
    }
    return server;
  }
  
  // makeStub: creates a stub for the remote network object represented
  // by wrep.  The name of the class for stub is stubClassName.
  ConcreteNetObjStub makeStub(WireRep wrep) throws NetException {
    // We suppose the stub is local (it is in the CLASSPATH).  If
    // it isn't, the operations fails.  Sorry, we don't provide the dynamic
    // class loading of RMI.
    try {
      Class stubClass= Class.forName(wrep.stubClassName);
      ConcreteNetObjStub stub= (ConcreteNetObjStub)stubClass.newInstance();
      stub.init(wrep, getServerConnection(wrep.sp));
      putInObjTbl(stub);
      return stub;
    }
    catch (Exception excp) {
      System.err.println("Manager: Can't make stub for class "+
                         wrep.stubClassName);
      throw new NetException(excp);
    }
  }

  // receiveNetObj: This method is invoked while deserializing a stream
  // if a wire representation is found.  It serves to translate the
  // the wire rep by the equivalent network object (its stub or concrete
  // object).
  //               *** Warning ***
  // This procedure is not synchronized to avoid the latencies of
  // its operations.
  Object receiveNetObj(WireRep wrep) throws NetException {
    NetObjBase netObj;
    ServerConnection server;
    Class stubClass;

    netObj= getInObjTbl(wrep);
    if (netObj!=null) {
      Debug.prt("Manager: receiving "+wrep+".  Found in object table.");
      return netObj; // The stub or the concrete representation for wrep was
                     // found in the object table.
    }

    // We have to create a stub for wrep
    ConcreteNetObjStub stub= makeStub(wrep);
    distGc.sendDirtyCall(wrep);
    return stub;
  }

  // sendNetObj: this procedure is invoked while serializing objects when
  // a network object is found.  This method ensures that all concrete
  // network objects having remote references have a skeleton, have
  // a wire representation and are present in the object table and
  // the own table.

  // The serial number for tagging concrete network objects
  private long serial= 1; // The 0 is for the local registry

  synchronized void sendNetObj(NetObjBase netObj) {
    // checks if the object is already present in the object table
    if (netObj.wrep!=null && getInObjTbl(netObj.wrep)!=null)
      return;

    // It is a concrete network object.
    ConcreteNetObj concNetObj= (ConcreteNetObj)netObj;

    // Check if the object has never been sent remotely
    if (concNetObj.wrep==null) {
      // Then, assigns a wire rep. for the object
      String stubClassName=
             chooseNarrowClass(concNetObj.getClass(),"Stub").getName();
      concNetObj.wrep= new WireRep(selfSp, serial++, stubClassName);
      Debug.prt("Manager: assigning wire rep. and skeleton for "+
                         concNetObj.wrep);

      // and its skeleton
      Class skelClass= chooseNarrowClass(concNetObj.getClass(), "Skel");
      // And a skeleton for dispatching the remote calls.
      try {
        concNetObj.skel= (ConcreteNetObjSkel)skelClass.newInstance();
      }
      catch (Exception excp) {
        System.err.println("Manager: couldn't create the skeleton for "+ 
                           concNetObj.getClass());
        // If this exception occurs, it is a bug
      }
    }

    // Puts the object in the object table.
    putInObjTbl(netObj);    // put it in the object table
  }

  // chooseNarrowClass: choose the best (narrowest) skeleton class apropiated
  // for class.  Should be invoked from synchronized methods only.
  private Class chooseNarrowClass(Class clazz, String suffix) {
    for(;;) {
      // Look for class in tbl
      String target= clazz.getName()+suffix;
      try {
        Class stubOrSkel= Class.forName(target);
        return stubOrSkel;
      }
      catch (ClassNotFoundException excp) {
      }
      // If not, try with the super class of class
      clazz= clazz.getSuperclass();
    }
  }

  // finalizeStub: invoked when `stub' is no more reachable on this JVM.
  // It removes the stub from the object table and informs the
  // distributed GC.  The distributed GC should send a clean call.
  //
  // Warning: A race condition may occur after the stub has been detected
  // non reachable (and therefore its weak reference has been cleared
  // in the object table) when a reference of the same object is received
  // before the finalization of the stub.  In such a case no clean call
  // should be sent.
  void finalizeStub(ConcreteNetObjStub stub) {
    if (cleanObjTbl(stub))
      distGc.sendCleanCall(stub.wrep);
  }

  //***************************************************************
  // Importing and exporting objects

  ConcreteRegistry registry= null;

  // setSocketPort: specifies which socket port to use for accepting
  // connections.
  public static void setSocketPort(int socketPort) throws NetException {
    synchronized (manager) {
      if (manager.serverSocket!=null)
        throw new Error("Manager: unable to set socket port because "+
           "the socket server was already created\n"+
           "Probably some network objects were bound or looked up before.");
      manager.socketPort= socketPort;
      manager.openServerSocket();
    }
  }

  // bind: export (or re-export) the network object netobj.  The object
  // will be named as objname.
  public static void bind(String objname, NetObj netObj) throws NetException {
    synchronized (manager) {
      // The server socket must be working
      if (manager.serverSocket==null)
        manager.openServerSocket();

      // and the registry be created
      if (manager.registry==null)
        manager.registry= new ConcreteRegistry();
    }
    manager.registry.bind(objname, netObj);
  }

  // getRegistry: get the remote registry of a process running on `host'
  // and listening to port `port'.
  public static Registry getRegistry(String host, int port)
    throws NetException, UnknownHostException {
    // The server socket must be working
    synchronized (manager) {
      if (manager.serverSocket==null)
        manager.openServerSocket();
    }

    // Don't synchronize because InetAddress.getByName could take some time
    Space sp= new Space(InetAddress.getByName(host), port);
    WireRep wrep= new WireRep(sp, 0, "netobj.ConcreteRegistryStub");
    return (Registry)manager.receiveNetObj(wrep);
  }
    
  // lookup: get a reference of a named remote network object.  The
  // string urladdr must be `netobj://<host>:<port>/<objname>'.
  public static NetObj lookup(String urladdr)
    throws MalformedURLException, NetException, UnknownHostException {
    URL url= new URL(urladdr);
    String host= url.getHost();
    int port= url.getPort();
    String objname= url.getFile().substring(1);
    Registry remoteRegistry= getRegistry(host, port);
    return remoteRegistry.lookup(objname);
  }

  //***************************************************************
  // startClientConnectionServer: it launches a thread that creates
  // a server socket for accepting connections to the local JVM

  ServerSocket serverSocket= null;

  synchronized void openServerSocket() {
    Debug.prt("Manager: launching socket server thread");
    try {
      serverSocket= new ServerSocket(socketPort);
      // The identification of the local JVM is constructed from
      // the IP number used for setting up the socket and the port
      // number found available for the server socket
      selfSp= new Space(InetAddress.getLocalHost(),
                        serverSocket.getLocalPort());
      Debug.prt("Manager: Local space identifier is "+selfSp);
    }
    catch (IOException excp) {
      throw new Error("Manager: I/O error while opening the server socket");
    }
    // Create the GC object
    distGc= new DistGC(selfSp, this);
    // starts a thread for accepting connections to the server socket
    Thread socketServerThread= new Thread(
      new Runnable() {
        public void run() {
          try {
            for(;;) {
              Debug.prt("Manager: Waiting for a socket request");
              Socket socket= serverSocket.accept(); // throws IOException
              ObjectOutputStream out=
                        new ObjectOutputStream(socket.getOutputStream()); 
              ObjectInputStream in=
                        new ObjectInputStream(socket.getInputStream());
              ConnectionFactory factory= (ConnectionFactory)in.readObject();
              factory.makeConnection(socket, in, out);
            }
          }
          catch (IOException excp) {
            System.err.println("ClientConnection (server): I/O error "+
              "while accepting a new connection");
          }
          catch (ClassNotFoundException excp) {
            throw new Error("Manager: buggy ClassNotFoundException");
          }
        }
      }
    );
    // This thread should never be collected by the GC
    socketServerThread.start();
    // Returns without waiting the later thread to finish
  }
}
