package netobj;

import java.io.*;
import java.net.*;

// The class ClientConnection represents connections to a client JVM.
// It is used to execute remote invocations of objects owned by the local
// JVM and issued by the client JVM.
public class ClientConnection implements Runnable {
  private Manager manager= Manager.getLocalManager();

  // The streams for communicating to the remote JVM
  private Socket socket;
          // The socket for communicating with the client JVM
  private ObjectInputStream in;   // to read from the socket
  private ObjectOutputStream out; // to write to the socket

  private Thread dispatcherThread;
          // The thread for dispatching remote invocations

  // Constructor: client connections are created by a unique thread
  // called the client connection server (launched by the manager).
  ClientConnection(Socket socket,
                   ObjectInputStream in, ObjectOutputStream out) {
    this.socket= socket;
    this.in= in;
    this.out= out;
    // The socket connection is open in the background by the
    // dispatcher thread
    Debug.prt("ClientConnection: Creating a client connection");
    dispatcherThread= new Thread(this);
    dispatcherThread.start();
  }

  // The body of the dispatcher thread.
  // It it finds any communication error, it closes the socket.
  // A new socket connection will be requested in the future by
  // the client JVM.
  public void run() {
   try {
      // loops reading remote requests of method invocations
      for(;;) {
        ConcreteNetObj netObj= (ConcreteNetObj)in.readObject();
          // throws OptionalDataException, ClassNotFoundException, IOExcp.
        int methodId= in.readInt(); // throws id.
        Debug.prt("ClientConnection: Dispatching a remote call for "+
                           "object "+netObj.wrep+", methodId "+methodId);

        // The local invocation of the remote call
        netObj.skel.dispatch(in, this, netObj, methodId); // throws IOExcp.
          // The skeleton must decode the parameters, invoke the method.
          // Then, if the call succedes, the skeleton must send back a
          // true value, followed by the returned value.  If the call fails,
          // the skeleton sends back a false value followed by the exception
          // that caused the failure.
        Debug.prt("ClientConnection: Finished remote call for "+
                           "object "+netObj.wrep.id+", methodId "+methodId);
      }
    }
    catch (EOFException excp) {
      // Somebody has closed the socket.  Do nothing.
    }
    // This following exceptions will happen due to serialization or socket
    // errors, not because the call failed.
    catch (OptionalDataException excp) {
      System.err.println("ClientConnection: serialization error while "+
                         "receiving data");    
    }
    catch (ClassNotFoundException excp) {
      System.err.println("ClientConnection: class not found while "+
                         "receiving data");    
    }
    catch (IOException excp) {
      System.err.println("ClientConnection: I/O error");
    }
    // The client JVM should initiate another connection in the future
    try { socket.close(); }
    catch (IOException excp) { }
    // The thread finishes
  }

  public void remoteRetVoid() throws IOException {
    out.writeLong(ServerConnection.TAG);
    out.writeBoolean(true);
    out.reset();
    out.flush();
  }

  public void remoteRetInt(int retInt) throws IOException {
    out.writeLong(ServerConnection.TAG);
    out.writeBoolean(true);
    out.writeInt(retInt);
    out.reset();
    out.flush();
  }

  public void remoteRetDouble(double retDouble) throws IOException {
    out.writeLong(ServerConnection.TAG);
    out.writeBoolean(true);
    out.writeDouble(retDouble);
    out.reset();
    out.flush();
  }

  public void remoteRetBoolean(boolean retBoolean) throws IOException {
    out.writeLong(ServerConnection.TAG);
    out.writeBoolean(true);
    out.writeBoolean(retBoolean);
    out.reset();
    out.flush();
  }

  public void remoteRetObject(Object retObj) throws IOException {
    out.writeLong(ServerConnection.TAG);
    out.writeBoolean(true);
    out.writeObject(retObj);
    out.reset();
    out.flush();
  }

  public void remoteRetException(Exception excp) throws IOException {
    out.writeLong(ServerConnection.TAG);
    out.writeBoolean(false);
    out.writeObject(excp);
    out.reset();
    out.flush();
  }

  public void checkTag() throws IOException {
    if (in.readLong()!=ServerConnection.TAG) {
        System.err.println("ClientConnection: "+
                           "did not get the expected tag");
        throw new IOException("ClientConnecton.checkTag: "+
                           "did not get the expected tag");
    }
  }
}
