#include <stdio.h>
#include <stdlib.h>

// This program generates the ucode of Piped LV32IM. This ucode contains the
// u-program that interpretes a RiscV instruction set.  It is stored in a ROM
// inside the control unit module and stores 128 u-instructions with
// the control signals to execute the gcc generated instructions for a
// Risc-V of 32 bits with mul/div extension.
// 
// Each u-instruction of the ucode ROM is 32 bit wide with the following format:
//
// bit 20 (decode): control signal to decode, which means that the next
//         u-instrution to execute is the one at u-address pointed by
//         the opcode in FInst
// bit 15 (ldst): control signal for load store memory operation,
//         with the type of load/store encoded in the func3 field of FInst
// bit 14 (alu): control signal for an arithmetic/logic operation,
//         with the operation encode in the func3 and func7 fields of FInst
// bit 12 (imm): control signal for an ALU operation with inmmediate operand
// etc.
// bits 6-0: For any u-instruction not decoding, it is the u-address of the
//         u-instruction to be executed in the next cycle
//

// Control signals: constants with CS preffix

// Type: define the ways of selecting the ALU Y operand
#define CSUTYPE    (1U<<29)
#define CSBTYPE    (2U<<29)
#define CSSTYPE    (3U<<29)
#define CSITYPE    (4U<<29)
#define CSJTYPE    (5U<<29)
#define CSY4       (6U<<29)
#define CSY0       (7U<<29)

#define OPTYPEMASK (7<<29)

// Single bit control signals needed by the CPU datapath (Piped Lrv32im module)

#define CSSELPC    (1<<0)
#define CSX0       (1<<1)
#define CSIMM      (1<<2)
#define CSWRRD     (1<<3)
#define CSALU      (1<<4)
#define CSLDST     (1<<5)
#define CSSELALU   (1<<6)
#define CSSELMEM   (1<<7)
#define CSWR       (1<<8)
#define CSRD       (1<<9)
#define CSBRANCH   (1<<10)
#define CSJMP      (1<<11)
#define CSDECODE   (1<<12)
#define CSFETCH    (1<<13)
#define CSSELZ     (1<<14)

// Control signals needed internally by the control unit (Control Unit module)

#define SIZEROM 128

// A uInst is a u-instruction in the ucode ROM
typedef unsigned int uInst;

// A uAddress is an address of a u-instruction in the ucode ROM
typedef int uAddress;

void dumpUcode();

uInst ucode[SIZEROM];        // The microcode ROM content
int endflag= 1;

char *opnames[SIZEROM];      // Names of Risc-V instructions
#define OPCODEBEGIN 0
#define OPCODEEND (SIZEROM-1)

uAddress upc= 0;    // The u-program counter, u-address of next u-instruction

// Defines the Risc-V instruction opname
void defInst(char *opname, int opcode) {
  if (opcode>OPCODEEND || opcode<0) {
    fprintf(stderr, "Opcode error for %s\n", opname);
    exit(1);
  }
  if (opnames[opcode]!=NULL) {
    fprintf(stderr, "Opcode for %s already used for %s\n",
                    opname, opnames[opcode]);
    exit(1);
  }
  upc= opcode;
  opnames[opcode]= opname;
}

// Add a new u-instruction at address upc.  The upc is incremented by 1.
// Parameter sig is a bit mask with control signals (bits 31-7)
void sig(uInst sig) {
  if (upc>=SIZEROM) {
    fprintf(stderr, "uROM size overflow\n");
    exit(1);
  }

  if (ucode[upc]!=0) {
    fprintf(stderr, "This u-instruction is already filled\n");
    exit(2);
  }

  ucode[upc++]= sig;
}

void cleanUcode() {
  upc= 0;
  while (upc<SIZEROM) {
    if (ucode[upc]!=0) {
      if (upc==SIZEROM-1 || ucode[upc+1]==0 || opnames[upc+1]!=NULL) {
        if (!(ucode[upc]&CSDECODE)) {
          fprintf(stderr, "u-instruction at %x doesn't finish with decode\n",
                          (unsigned)upc);
          exit(3);
        }
        if (!(ucode[upc]&CSFETCH)) {
          fprintf(stderr, "u-instruction at %x doesn't finish with fetch\n",
                          (unsigned)upc);
          exit(3);
        }
      }
    }
    upc++;
  }
}

int main() {
  // First u-instruction with u-address 0 (upc is 0)
  // Bootstrap: first Risc-V instruction must be at RAM address 0x0
                                            // AR <- 0, PC <- Mem[AR]
  sig(CSRD | CSFETCH);
  // FPC <- MPC, FINST <- M[MPC], MPC <- MPC+4,

  // LBPIPE: decode next instruction and fetch another instruction
  sig(CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4

  // Logic and arithmetic instructions

  defInst("reg_alu_inst", 0x33);
  sig(CSALU | CSWRRD | CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4,
  // Rd <- Rs1 func3 y func7 Rs2
  // The ALU operation is encoded in func3 with variations in func7

  defInst("imm_alu_inst", 0x13);
  sig(CSALU | CSIMM | CSITYPE | CSWRRD | CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4,
  // Rd <- Rs1 func3 I
  // The ALU operation is encoded in func3 (doesn't use func7)

  // Memory access

  defInst("load_inst", 0x03);
  sig(CSIMM | CSITYPE | CSSELALU | CSLDST | CSRD | CSSELMEM | CSWRRD);
  // Rd <- M[Rs1 + I]
  sig(CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4


  defInst("store_inst", 0x23);
  sig(CSIMM | CSSTYPE | CSSELALU | CSLDST | CSWR);
  // M[Rs1 + S] <- Rs2
  sig(CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4

  // Special instructions

  defInst("lui", 0x37); // load upper immediate
  sig(CSX0 | CSUTYPE | CSWRRD | CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4,
  // Rd <- U

  defInst("auipc", 0x17); // add upper immediate to pc
  sig(CSSELPC | CSUTYPE | CSWRRD | CSRD | CSFETCH | CSDECODE);
  // Rd <- PC + U,
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4

  // Call function and return

  defInst("jal", 0x6f); // jump and link
  sig(CSSELPC | CSY4 | CSWRRD);
  // Rd <- PC + 4
  sig(CSSELPC | CSJTYPE | CSSELALU | CSJMP | CSRD | CSFETCH);
  // MPC <- PC + J, FINST <- M[PC+J]
  sig(CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4


  defInst("jalr", 0x67); // jump and link register
  sig(CSSELPC | CSY4 | CSWRRD);
  // Rd <- PC + 4
  sig(CSITYPE | CSSELALU | CSJMP | CSRD | CSFETCH);
  // MPC <- Rs1 + J, FINST <- M[Rs1+J]
  sig(CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4

  defInst("branch", 0x63);
  sig(CSSELPC | CSBTYPE | CSBRANCH | CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST
  // if (taken) MPC <- PC + B, FINST <- M[PC+B], don't decode, goto next u-inst.
  // else MPC <- MPC+4, FINST <- M[MPC+4], do decode
  sig(CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4,
  
  defInst("ecall", 0x73);                   // not implemented
  sig(CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4,

  defInst("fence", 0x0f);                   // not implemented
  sig(CSRD | CSFETCH | CSDECODE);
  // PC <- FPC, FPC <- MPC, DINST <- FINST, FINST <- M[MPC], MPC <- MPC+4,

  // Add here the experimental instructions

  // defInst("blabla", opcode);
  // sig( ... );
  // ...
  // sig( ... );

  // =====================================
  // Beware: do not modify following lines
  // =====================================


  cleanUcode();

  dumpUcode();

  // Write opcodes
  FILE *ucodeROM= fopen("ucode.rom", "w");
  if (ucodeROM==NULL) {
    perror("ucode.rom");
    exit(1);
  }

  fprintf(ucodeROM,"v2.0 raw\n");
  for (int i= 0; i<SIZEROM; i++) {
    fprintf(ucodeROM,"%x\n", ucode[i]);
  }
  fclose(ucodeROM);

  return 0;
}

void prtCs(char *cs) {
  printf(" %s", cs);
}

void dumpUcode() {
  printf("Content of ucode.rom.  Each line corresponds to a u-instruction.\n");
  printf("The fields in the listing are:\n");
  printf("  <u-address>:\n");
  printf("  <u-instruction content>\n");
  printf("  <control signals>\n");
  printf("(all numbers are in hexadecimal notation)\n\n");
  printf("Bootstrap:\n");

  for (uAddress i= 0; i< SIZEROM; i++) {
    uInst sig= ucode[i];
    if (sig==0)
      continue;

    if (opnames[i]!=NULL)
      printf("Inst: %s\n", opnames[i]);

    printf("%02x: %08x ", i, sig); 

    if (CSSELPC & sig) prtCs("selpc");
    else if (CSX0 & sig) prtCs("x0");

    int type= sig & OPTYPEMASK;
    switch (type) {
      case CSY0: prtCs("y0"); break;
      case CSY4: prtCs("y4"); break;
      case CSUTYPE: prtCs("upper"); break;
      case CSBTYPE: prtCs("branch"); break;
      case CSSTYPE: prtCs("store"); break;
      case CSITYPE: prtCs("itype"); break;
      case CSJTYPE: prtCs("jal"); break;
      default: prtCs("reg"); break;
    }

    if (CSIMM & sig) prtCs("imm");
    if (CSWRRD & sig) prtCs("wrrd");
    if (CSWR & sig) prtCs("wr");
    if (CSRD & sig) prtCs("rd");
    if (CSBRANCH & sig) prtCs("branch");
    if (CSJMP & sig) prtCs("jmp");
    if (CSFETCH & sig) prtCs("fetch");
    if (CSSELMEM & sig) prtCs("selmem");
    if (CSSELALU & sig) prtCs("selalu");
    if (CSDECODE & sig) prtCs("decode");
    if (CSALU & sig) prtCs("alu");
    if (CSLDST & sig) prtCs("ldst");

    printf("\n");
  }
}
