#include <stdio.h>
#include <stdlib.h>

// This program generates the ucode of LV32i. This ucode contains the u-program 
// that interpretes a RiscV instruction set.  It is stored in a ROM
// inside the control unit (cu2 module) and stores 128 u-instructions with
// the control signals to execute the gcc generated instructions for a
// Risc-V of 32 bits with mul/div extension.
// 
//
// Each u-instruction of the ucode ROM is 32 bit wide with the following format:
//
// bits 31-9: control signals for the CPU datapath (cpu module)
// bit 8: control signal for fetch cycle
// bit 7: control signal to decode, which means that the next u-instruction
//        to execute is the one at u-address pointed by the opcode in IR
// bits 6-0: For any u-instruction not decoding IR, it is the u-address of the
//        u-instruction to be executed in the next cycle
//

// Control signals

// OPYSEL: define the ways of selecting the ALU Y operand
#define CSYREG     (0<<24)
#define CSYUPPER   (1<<24)
#define CSYBRANCH  (2<<24)
#define CSYSTORE   (3<<24)
#define CSYIMM     (4<<24)
#define CSYJAL     (5<<24)
#define CSY4       (6<<24)
#define CSY0       (7<<24)

// Single bit control signals needed by the CPU datapath (cpu module)
#define CSSELPC    (1<<27)
#define CSX0       (1<<28)
#define CSWRRD     (1<<23)
#define CSSWAPDS1  (1<<22)
#define CSWRPC     (1<<21)
#define CSBRANCH   (1<<20)
#define CSWRIR     (1<<14)
#define CSSELT     (1<<19)
#define CSSELD     (1<<15)
#define CSWRAR     (1<<16)
#define CSRD       (1<<17)
#define CSWR       (1<<18)
#define CSINCPC    (1<<9)
#define CSIMM      (1<<29)

// Control signals needed internally by the control unit (cu2 module)
#define CSDECODE   (1<<7)
#define CSADD      (1<<8)

// Mask for the u-address field of the u-instruction
#define UADDRMASK  0x7F
#define OPYSELMASK (7<<24)

#define SIZEROM 128

// A uInst is a u-instruction in the ucode ROM
typedef unsigned int uInst;

// A uAddress is an address of a u-instruction in the ucode ROM
typedef int uAddress;

// A uLabel tags the specific u-address of a u-instruction
// Needed by u-instructions that are the target of a u-jump.
typedef int uLabel;

void dumpUcode(uAddress i);

uInst ucode[SIZEROM];        // The microcode ROM content
int endflag= 1;
// uInst scode[SIZEROM];     // Unordered microcode
// uAddress uaddr[SIZEROM];  // u-instruction at scode[i] must be placed at
                             // ucode[uaddr[i]]

char *opnames[SIZEROM];      // Names of Risc-V instructions
#define OPCODEBEGIN 0
#define OPCODEEND (SIZEROM-1)

uAddress upc= 0;    // The u-instruction address
// uAddress spc= 0; // The unordered u-instruction addressº

// u-address of the fetch u-instruction
#define LBFETCH 1
#define LBDECODE 2

// Defines the Risc-V instruction opname
void defInst(char *opname, int opcode) {
  if (opcode>OPCODEEND || opcode<0) {
    fprintf(stderr, "Opcode error for %s\n", opname);
    exit(1);
  }
  if (opnames[opcode]!=NULL) {
    fprintf(stderr, "Opcode for %s already used for %s\n",
                    opname, opnames[opcode]);
    exit(1);
  }
  upc= opcode;
  opnames[opcode]= opname;
}

// Add a new u-instruction at address upc.  The upc is incremented by 1.
// Parameter sig is a bit mask with control signals (bits 31-7)
void sig(uInst sig) {
  if (upc>=SIZEROM) {
    fprintf(stderr, "uROM size overflow\n");
    exit(1);
  }

  if (ucode[upc]!=0) {
    fprintf(stderr, "This u-instruction is already filled\n");
    exit(1);
  }

  ucode[upc++]= sig;
}

void cleanUcode() {
  upc= 0;
  while (upc<SIZEROM) {
    if (ucode[upc]!=0) {
      if ((ucode[upc]&UADDRMASK)==0 && upc+1<SIZEROM &&
          ucode[upc+1]!=0 && opnames[upc+1]==NULL) {
        ucode[upc] |= upc+1;
      }
      else if (!(ucode[upc]&CSDECODE) && (ucode[upc]&UADDRMASK)==0) {
        ucode[upc] |= LBFETCH | CSINCPC;
      }
    }
    upc++;
  }
}

int main() {
  // First u-instruction with u-address 0 (upc is 0)
  // Bootstrap: address of first Risc-V instruction must be at RAM address 0x0
                                            // AR <- 0, PC <- Mem[AR]
  sig(CSRD | CSSELD | CSWRIR | CSWRPC);
  // CSWRIR forces control unit to drive LDW to data bus interface (DBI)

  // Fetch cycle:                              AR <- PC, IR <- Mem[AR]
  sig(CSSELPC | CSY0 | CSADD | CSWRAR | CSRD | CSWRIR);
  // CSADD forces control unit to drive ADD to ALU

  // Decode cycle
  sig(CSDECODE);                            // uPC <- opcode

  // Logic and arithmetic instructions

  defInst("reg_alu_inst", 0x33);
  sig(CSYREG | CSWRRD);                     // Rd <- Rs1 func3 Rs2
  // The ALU operation is encoded in func3 with variations in func7

  defInst("imm_alu_inst", 0x13);
  sig(CSIMM | CSYIMM | CSWRRD);             // Rd <- Rs1 func3 I
  // The ALU operation is encoded in func3 with variations in func7

  defInst("load_inst", 0x03);               // AR <- Rs1 + I, Rd <- Mem[AR]
  sig(CSYIMM | CSADD | CSWRAR | CSRD | CSSELD | CSWRRD);
  // The type of load is encoded in func3

  defInst("store_inst", 0x23);
  sig(CSYSTORE | CSADD | CSWRAR);           // AR <- Rs1 + S
  sig(CSX0 | CSYREG | CSADD | CSWR);        // Mem[AR] <- Rs2

  defInst("lui", 0x37);                     // load upper immediate
  sig(CSX0 | CSYUPPER | CSADD | CSWRRD);    // Rd <- U<<12

  defInst("auipc", 0x17);                   // add upper immediate to pc
  sig(CSSELPC | CSYUPPER | CSADD | CSWRRD); // Rd <- PC + (U<<12)

  defInst("jal", 0x6f);
  sig(CSSELPC | CSY4 | CSADD | CSWRRD);     // Rd <- PC
  // Modify PC and fetch next instruction      AR <- PC <- PC + J, IR <- Mem[AR]
  sig(CSSELPC | CSYJAL | CSADD | CSWRPC | CSWRAR | CSRD | CSWRIR | LBDECODE);

  defInst("jalr", 0x67);
  sig(CSSELPC | CSY4 | CSADD | CSWRRD);     // Rd <- PC
  // Modify PC and fetch next instruction   AR <- PC <- Rs1 + J, IR <- Mem[AR]
  sig(CSYIMM | CSADD | CSWRPC | CSWRAR | CSRD | CSWRIR | LBDECODE);

  defInst("cond_branch", 0x63);             // if taken PC <- PC + B
  sig(CSSELPC | CSYBRANCH | CSADD | CSBRANCH);

  defInst("ecall", 0x73);                   // not implemented
  sig(CSINCPC);

  defInst("fence", 0x07);                   // not implemented
  sig(CSINCPC);

  // Add here the experimental instructions

  // defInst("blabla", opcode);
  // sig( ... );
  // ...
  // sig( ... );

  defInst("swap", 0x9);
  // Ejemplo: s2, 8(sp)  # Intercambia el valor de s2 con el valor de Mem(sp+8)
  // DBI requires func3 to be 010                   AR <- Rs1 + I, T <- Mem[AR]
  sig(CSYIMM | CSADD | CSWRAR | CSRD | CSSELD | CSSELT | CSWRPC);
  sig(CSSWAPDS1 | CSY0 | CSADD | CSWR);      	 // Mem[AR] <- Rd
  sig(CSSELT | CSSELPC | CSY0 | CSADD | CSWRRD); // Rd <- T

  defInst("stwx2", 0x50);  // opcode for stwx2 is 0x50
  // Ejemplo: stwx2 a0, s4, (sp) # Mem[sp] <- a0, Mem[sp+4] <- s4
  // a0 es X10 y se especifica en el campo rd (bits 11-7 codifican 10)
  // s4 es X20 y se especifica en el campo rs2 (bits 24-20 codifican 20)
  // sp es X2 y se especifica en el campo rs1 (bits 19-15 codifican 2 )
  // 
  sig(CSINCPC);  // Change this!  This is just to skip this instruction
  // sig( ... );    You can implement stwx2 in 4 u-instructions!
  // sig( ... );
  // sig( ... );

  // =====================================
  // Beware: do not modify following lines
  // =====================================


  cleanUcode();

  // Write opcodes
  FILE *ucodeROM= fopen("ucode.rom", "w");
  if (ucodeROM==NULL) {
    perror("ucode.rom");
    exit(1);
  }

  printf("Content of ucode.rom.  Each line corresponds to a u-instruction.\n");
  printf("The fields in the listing are:\n");
  printf("  <u-address>:\n");
  printf("  <u-instruction content>\n");
  printf("  (<address of next u-instruction to be executed>)\n");
  printf("  <control signals>\n");
  printf("  / Inst:<name of implemented instruction> (opt. field)\n");
  printf("(all numbers are in hexadecimal notation)\n\n");
  fprintf(ucodeROM,"v2.0 raw\n");
  for (int i= 0; i<SIZEROM; i++) {
    dumpUcode(i);
    fprintf(ucodeROM,"%x\n", ucode[i]);
  }
  fclose(ucodeROM);

  return 0;
}

void prtCs(char *cs) {
  printf(" %s", cs);
}

void dumpUcode(uAddress i) {
  uInst sig= ucode[i];
  if (sig==0)
    return;

  printf("%02x: %08x (%02x)", i, sig, sig & UADDRMASK); 

  if (CSSELPC & sig) prtCs("selpc");
  else if (CSX0 & sig) prtCs("x0");

  int opysel= sig & OPYSELMASK;
  switch (opysel) {
    case CSY0: prtCs("y0"); break;
    case CSY4: prtCs("y4"); break;
    case CSYREG: prtCs("yreg"); break;
    case CSYUPPER: prtCs("yupper"); break;
    case CSYBRANCH: prtCs("ybranch"); break;
    case CSYSTORE: prtCs("ystore"); break;
    case CSYIMM: prtCs("yimm"); break;
    case CSYJAL: prtCs("yjal"); break;
  }

  if (CSWRRD & sig) prtCs("wrrd");
  if (CSWRPC & sig) prtCs("wrpc");
  if (CSWRIR & sig) prtCs("wrir");
  if (CSWRAR & sig) prtCs("wrar");
  if (CSSELD & sig) prtCs("seld");
  if (CSSWAPDS1 & sig) prtCs("swapds1");
  if (CSBRANCH & sig) prtCs("branch");
  if (CSRD & sig) prtCs("RD");
  if (CSWR & sig) prtCs("WR");
  if (CSDECODE & sig) prtCs("dec");
  if (CSINCPC & sig) prtCs("incpc");
  if (CSIMM & sig) prtCs("imm");
  if (CSADD & sig) prtCs("add");
  if (CSSELT & sig) prtCs("selt");

  if (opnames[i]!=NULL)
    printf(" / Inst: %s", opnames[i]);
  printf("\n");
}
