#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "asm.h"

#define MINOP 1
#define MAXOP 31
#define MAXIMM4  ((1<<3)-1)
#define MINIMM4  (-MAXIMM4-1)
#define MAXIMM5  ((1<<4)-1)
#define MINIMM5  (-MAXIMM5-1)

// Branches
#define BE      0
#define BNE     (1<<8)
#define BA      (2<<8)

#define BG      (4<<8)
#define BGE     (5<<8)
#define BL      (6<<8)
#define BLE     (7<<8)

#define SIZEROMW (SIZEROM/2)

Reg reg0= {0};
Reg reg1= {1};
Reg reg2= {2};
Reg reg3= {3};
Reg reg4= {4};
Reg reg5= {5};
Reg reg6= {6};
Reg reg7= {7};

static unsigned short rom[SIZEROMW];
static char iscode[SIZEROMW];

static int pc= 0;

static int lab4(Label label);
static int lab16(Label label);
static void checkLabels();
static void listing();

static void gen(int inst) {
  iscode[pc]= 1;
  rom[pc++]= inst;
}

static int opcode(int op) {
  if (! (MINOP<=op && op<=MAXOP)) {
    fprintf(stderr, "Invalid opcode %d\n", op);
    exit(1);
  }
  return op<<11;
}

static int unary(int op2) {
  if (! (0<=op2 && op2<=7)) {
    fprintf(stderr, "Invalid unary op %d\n", op2);
    exit(1);
  }
  return op2<<5;
}

static void sethi(int val, int sgnpos) {
  if (val&(1<<(sgnpos-1)))
    val= ~val;
  gen(opcode(OPsethi) | ((val>>4) & 0x0fff));
}

static int regd(Reg *preg) {
  int r= preg->r;
  if (! (0<=r && r<=7)) {
    fprintf(stderr, "Invalid reg %d\n", r);
    exit(1);
  }
  return r<<8;
}

static int regs1(Reg *preg) {
  int r= preg->r;
  if (! (0<=r && r<=7)) {
    fprintf(stderr, "Invalid reg %d\n", r);
    exit(1);
  }
  return r<<5;
}

static int regs2(Reg *preg, int sh) {
  int r= preg->r;
  if (! (0<=r && r<=7)) {
    fprintf(stderr, "Invalid reg %d\n", r);
    exit(1);
  }
  return (r<<1) | (sh ? 1 : 0);
}

static int imm4(int val) {
  if (! (MINIMM4<=val && val<=MAXIMM4))
    sethi(val, 4);
  return val & ((1<<4)-1);
}

static int imm5(int val) {
  if (! (MINIMM5<=val && val<=MAXIMM5))
    sethi(val, 5);
  return val & ((1<<5)-1);
}

static void genreg(int op, Reg *rs1, Reg *rs2, Reg *rd) {
  gen(opcode(op) | regs1(rs1) | regs2(rs2,0) | regd(rd));
}
  
static void genimm4(int op, Reg *rs1, int val, Reg *rd) {
  gen(opcode(op) | regs1(rs1) | 0x10 | imm4(val) | regd(rd));
}
  
void mov(Reg *rs2, Reg *rd) {
  gen(unary(UNARYmov) | regs2(rs2,0) | regd(rd));
}

void movi(int val, Reg *rd) {
  gen(unary(UNARYmov) | 0x10 | imm4(val) | regd(rd));
}

void mova(Label label, Reg *rd) {
  gen(unary(UNARYmov) | 0x10 | lab4(label) | regd(rd));
}

void neg(Reg *rs2, Reg *rd) {
  gen(unary(UNARYneg) | regs2(rs2,0) | regd(rd));
}

void call(Label label) {
  callr(label, R0);
}

void callr(Label label, Reg *rd) {
  gen(unary(UNARYcallr) | regd(rd));
  gen(lab16(label));
}

void ret() {
  jmpr(R0);
}

void jmpr(Reg *rs2) {
  gen(unary(UNARYjmpr) | regs2(rs2, 0));
}

void jmpl(Reg *rd, Reg *rs2) {
  gen(unary(UNARYjmpl) | regd(rd) | regs2(rs2, 0));
}

void add(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPadd, rs1, rs2, rd);
}
  
void addi(Reg *rs1, int val, Reg *rd) {
  genimm4(OPadd, rs1, val, rd);
}

void sub(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPsub, rs1, rs2, rd);
}
  
void subi(Reg *rs1, int val, Reg *rd) {
  genimm4(OPsub, rs1, val, rd);
}

void shl(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPshl, rs1, rs2, rd);
}

void shli(Reg *rs1, int val, Reg *rd) {
  genimm4(OPshl, rs1, val, rd);
}

void shr(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPshr, rs1, rs2, rd);
}

void shri(Reg *rs1, int val, Reg *rd) {
  genimm4(OPshr, rs1, val, rd);
}

void sar(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPsar, rs1, rs2, rd);
}

void sari(Reg *rs1, int val, Reg *rd) {
  genimm4(OPsar, rs1, val, rd);
}

void and(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPand, rs1, rs2, rd);
}

void andi(Reg *rs1, int val, Reg *rd) {
  genimm4(OPand, rs1, val, rd);
}

void or(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPor, rs1, rs2, rd);
}

void ori(Reg *rs1, int val, Reg *rd) {
  genimm4(OPor, rs1, val, rd);
}

void xor(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPxor, rs1, rs2, rd);
}

void xori(Reg *rs1, int val, Reg *rd) {
  genimm4(OPxor, rs1, val, rd);
}

void mul(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPmul, rs1, rs2, rd);
}

void muli(Reg *rs1, int val, Reg *rd) {
  genimm4(OPmul, rs1, val, rd);
}

void divr(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPdiv, rs1, rs2, rd);
}

void udivi(Reg *rs1, int val, Reg *rd) {
  genimm4(OPdiv, rs1, val, rd);
}

void rem(Reg *rs1, Reg *rs2, Reg *rd) {
  genreg(OPrem, rs1, rs2, rd);
}

void remi(Reg *rs1, int val, Reg *rd) {
  genimm4(OPrem, rs1, val, rd);
}

void ldsb(Reg *ra, int idx, Reg *rd) {
  gen(opcode(OPldsb) | regd(rd) | regs1(ra) | imm5(idx));
}

void ldub(Reg *ra, int idx, Reg *rd) {
  gen(opcode(OPldub) | regd(rd) | regs1(ra) | imm5(idx));
}

void ldw(Reg *ra, int idx, Reg *rd) {
  gen(opcode(OPldw) | regd(rd) | regs1(ra) | imm5(idx));
}

void stb(Reg *rs, Reg *ra, int idx) {
  gen(opcode(OPstb) | regd(rs) | regs1(ra) | imm5(idx));
}

void stw(Reg *rs, Reg *ra, int idx) {
  gen(opcode(OPstw) | regd(rs) | regs1(ra) | imm5(idx));
}

void ba(Label label) {
  gen(opcode(OPbranchs) | BA);
  gen(lab16(label));
}

void be(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchs) | BE | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void bne(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchs) | BNE | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void bei(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchs) | BE | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

void bnei(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchs) | BNE | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

void bg(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchs) | BG | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void bge(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchs) | BGE | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void bl(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchs) | BL | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void ble(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchs) | BLE | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void bgu(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchu) | BG | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void bgeu(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchu) | BGE | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void blu(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchu) | BL | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void bleu(Reg *rs1, Reg *rs2, Label label) {
  gen(opcode(OPbranchu) | BLE | regs1(rs1) | regs2(rs2, 0));
  gen(lab16(label));
}

void bgi(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchs) | BG | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

void bgei(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchs) | BGE | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

void bli(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchs) | BL | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

void blei(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchs) | BLE | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

void bgui(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchu) | BG | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

void bgeui(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchu) | BGE | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

void blui(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchu) | BL | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

void bleui(Reg *rs1, int val, Label label) {
  gen(opcode(OPbranchu) | BLE | regs1(rs1) | 0x10 | imm4(val));
  gen(lab16(label));
}

static FILE *romfile;

void init(int argc, char *argv[]) {
  if (argc!=2) {
    fprintf(stderr, "Usage: %s <rom-file>\n", argv[0]);
    exit(1);
  }
  char *romname= argv[1];

  romfile= fopen(romname, "w");
  if (romfile==NULL) {
    perror(romname);
    exit(1);
  }

  rom[SIZEROMW-1]= ROMINI; // 0xf000
  pc= -1;
  Label term= getLabel("TERM");
  setLabel(term); // Terminal port
  pc= 0;
  movi(SIZERAM, R7); // Initializes stack
}

void finish() {
  checkLabels();
  listing();
  fprintf(romfile, "v2.0 raw\n");
  for (int i= 0; i<pc; i++)
    fprintf(romfile, "%x\n", rom[i]);
  fprintf(romfile,"%d*0\n", (SIZEROMW-pc-1));
  fprintf(romfile,"%04x\n", rom[SIZEROMW-1]);
  fclose(romfile);
}

//===========================
// R/O data
//===========================

void string(char *str) {
  int len= strlen(str);
  if (pc+(len+2)/2>SIZEROMW) {
    fprintf(stderr, "ROM size exceeded\n");
    exit(1);
  }
  strcpy((char*)&rom[pc], str);
  pc += (len+2)/2;
}

void word(short w) {
  rom[pc++]= w;
}

void uword(unsigned short uw) {
  rom[pc++]= uw;
}

//===========================
// Label management
//===========================

static Label labelList= NULL;

#define NULLIDX 0x80000000

Label getLabel(char *name) {
  Label label= labelList;
  while (label!=NULL) {
    if (strcmp(label->name, name)==0)
      return label;
    label= label->next;
  }
  label= malloc(sizeof(*label));
  label->idx= NULLIDX;
  label->name= name;
  label->pend= NULL;
  label->next= labelList;
  labelList= label;
  return label;
}

static void addPend(Label label, int idx, int size) {
  Node *node= malloc(sizeof(Node));
  node->idx= idx;
  node->size= size; // 4 or 16
  node->next= label->pend;
  label->pend= node;
}

void setLabel(Label label) {
  label->idx= pc;
  for (Node *node= label->pend; node!=NULL; node= node->next) {
    if (node->size==16) { // This is for branches or calls
      rom[node->idx] = pc*2 + ROMINI;
    }
    else if (node->size==4) { // This is for a inmmediate
      int val= pc;
      int savepc= pc;         // Use sethi always
      pc= node->idx;
      val = val*2 + ROMINI;
      sethi(val, 4);
      pc= savepc;
      rom[node->idx+1] |= val&0xf;
    }
  }
}

static void checkLabels() {
  for (Label label= labelList; label!=NULL; label= label->next) {
    if (label->idx==NULLIDX) {
      fprintf(stderr, "Label %s was not defined\n", label->name);
      exit(1);
    }
  }
}

static void printNames(int idx) {
  for (Label label= labelList; label!=NULL; label= label->next) {
    if (label->idx==idx)
      printf("            %s:\n",label->name);
  }
}

static char *getLabelName(int idx) {
  for (Label label= labelList; label!=NULL; label= label->next) {
    if (label->idx==idx)
      return label->name;
  }
  return NULL;
}

static int lab4(Label label) {
  if (label->idx>=0)
    return imm4((label->idx)*2+ROMINI);
  else {
    addPend(label, pc, 4);
    return imm4(0x7ff0);
  }
}

static int lab16(Label label) {
  if (label->idx>=0)
    return label->idx*2 + ROMINI;
  else {
    addPend(label, pc, 16);
    return 0;
  }
}

//======================================
// Listing : disassembles generated code
//======================================

static char *ops[32];
static char *branches[8]={"be", "bne", "ba", NULL, "bg", "bge", "bl", "be"};

static void listreg(int r) {
  printf("R%d", r);
}

static void listval(int val) {
  printf("%d", val);
}

static void listaddr(unsigned short addr) {
  int idx= (addr-ROMINI)/2;
  printf("%04x", addr);
  char *name= getLabelName(idx);
  if (name!=NULL)
    printf("   (%s)", name);
}

static int prefix;

static int decimm(int inst, int sgn) {
  int hival= prefix;
  if (inst & (1<<(sgn-1)))
    hival = ~hival;
  hival <<= 4;
  int masklo= (1<<sgn)-1;
  int maskhi= ~masklo;
  short val= (inst & masklo) | (hival & maskhi);
  return val;
}

static void listing() {
  prefix= 0;
  ops[OPadd]= "add";
  ops[OPsub]= "sub";
  ops[OPshl]= "shl";
  ops[OPshr]= "shr";
  ops[OPsar]= "sar";
  ops[OPand]= "and";
  ops[OPor]= "or";
  ops[OPxor]= "xor";
  ops[OPmul]= "mul";
  ops[OPdiv]= "div";
  ops[OPrem]= "rem";
  ops[OPldw]= "ldw";
  ops[OPldwpp]= "ldwpp";
  ops[OPldub]= "ldub";
  ops[OPldsb]= "ldsb";
  ops[OPstw]= "stw";
#ifdef OPswap
  ops[OPswap]= "swap";
#endif
  ops[OPstb]= "stb";
  ops[OPsethi]= "sethi";
  ops[OPsethi_dummy]= "sethi";
  for (int idx= 0; idx<pc; idx++) {
    printNames(idx);
    printf("%04x: %04x      ", idx*2+0xf000, rom[idx]);
    if (iscode[idx]) {
      int inst= rom[idx];
      int op= inst>>11;
      int rd= (inst>>8) & 7;
      int rs1= (inst>>5) & 7;
      int rs2= (inst>>1) & 7;
      int imm4= decimm(inst, 4);
      int imm5= decimm(inst, 5);
      int isval= inst & (1<<4);
      if (op==0) {
        switch(rs1) {
          case UNARYmov:
          case UNARYneg:
            printf("%s ", rs1==UNARYmov ? "mov" : "neg");
            if (isval)
              listval(imm4);
            else
              listreg(rs2);
            printf(", ");
            listreg(rd);
            if (isval && rs1==UNARYmov) {
              char *name= getLabelName(((unsigned short)imm4-ROMINI)/2);
              if (name!=NULL)
                printf("  (probably %s)", name);
            }
            break;
          case UNARYjmpl:
            printf("jmpl ");
            listreg(rs2);
            printf(", ");
            listreg(rd);
            break;
          case UNARYcallr:
            printf(rd==0 ? "call " : "callr ");
            listaddr(rom[++idx]);
            if (rd!=0) {
              printf(", ");
              listreg(rd);
            }
            printf("\n%04x: %04x", idx*2, rom[idx]);
            break;
          case UNARYjmpr:
            if (!isval && rs2==0) {
              printf("ret");
              break;
            }
            printf("jmpr ");
            listreg(rs2);
            break;
        }
      }
      else if (op==OPbranchu || op==OPbranchs) {
        int addr= rom[++idx];
        printf("%s%c ", branches[rd], op==OPbranchs ? ' ' : 'u');
        if (rd!=2) { // not ba
          listreg(rs1);
          printf(", ");
          if (isval)
            listval(imm4);
          else
            listreg(rs2);
          printf(", ");
        }
        listaddr(addr);
        printf("\n%04x: %04x", idx*2, addr);
      }
      else if (op<OPsethi) {
        printf("%s ", ops[op]);
        switch(op) {
          case OPadd:
          case OPsub:
          case OPshl:
          case OPshr:
          case OPsar:
          case OPand:
          case OPor:
          case OPxor:
          case OPmul:
          case OPdiv:
          case OPrem:
            listreg(rs1); printf(", ");
            if (isval)
              listval(imm4);
            else
              listreg(rs2);
            printf(", ");
            listreg(rd);
            break;
          case OPldwpp:
          case OPldw:
          case OPldub:
          case OPldsb:
            listreg(rs1);
            printf("[");
            listval(imm5);
            printf("], ");
            listreg(rd);
            break;
#ifdef OPswap
          case OPswap:
#endif
          case OPstw:
          case OPstb:
            listreg(rd);
            printf(", ");
            listreg(rs1);
            printf("[");
            listval(imm5);
            printf("]");
            break;
        }
      }
      else if (op==OPsethi || op==OPsethi_dummy) {
        switch(op) {
          case OPsethi:
          case OPsethi_dummy:
            prefix= inst & 0x0fff;
            printf("sethi");
            break;
        }
      }
      else {
        fprintf(stderr," Invalid opcode: %d\n", op);
        exit(1);
      }
      if (op!=OPsethi && op!=OPsethi_dummy)
        prefix= 0;
    }
    printf("\n");
  }
}

//===========================
// Utility functions
//===========================

static int printStrGenerated= 0;

Label printStrGen() {
  Label printStr= getLabel("printStr");
  if (printStrGenerated)
    return printStr;

  Label loop= getLabel("printStr$loop");
  Label end= getLabel("printStr$end");

  setLabel(printStr); // printStr:         # void printStr(char*s) {
  movi(TERM, R2);     //   mov TERM, R2    #   char *term= (char*)TERM;
  setLabel(loop);     // loop:             #   for (;;) {
  ldsb(R1, 0, R3);    //   ldsb R1[0], R3  #     int c= *s;
  bei(R3, 0, end);    //   be R3, 0, end   #     if (c==0) break;
  stb(R3, R2, 0);     //   stb R3, R2[0]   #     *term= c;
  addi(R1, 1, R1);    //   addi R1, 1, R1  #     s++;
  ba(loop);           //   ba loop         #   }
  setLabel(end);      // end:              #
  ret();              //   ret             # }

  printStrGenerated= 1;
  return printStr;
}

static int getcharGenerated= 0;

Label getcharGen() {
  Label getchar= getLabel("getchar");
  if (getcharGenerated)
    return getchar;

  Label loop= getLabel("getchar$loop");

  setLabel(getchar);  // getchar:          # int getchar() {
  movi(TERM, R2);     //   mov TERM, R2    #   char c, *term= (char*)TERM;
  setLabel(loop);     // loop:             #   do {
  ldsb(R2, 0, R1);    //   ldsb R2, 0, R1  #     c= *term;
  bli(R1, 0, loop);   //   bl R1, 0, loop  #   } while (c<0);
  stb(R1, R2, 0);     //   stb R1, R2[0]   #   *term= c;
  ret();              //   ret             #   return c;

  getcharGenerated= 1; //                  # }
  return getchar;
}

static int readUintGenerated= 0;

Label readUintGen() {
  Label readUint= getLabel("readUint");
  if (readUintGenerated)
    return readUint;

  Label loop= getLabel("readUint$loop");
  Label end= getLabel("readUint$end");
  Label getchar= getLabel("getchar");

  setLabel(readUint); // readUint:         # int readUint() {
  subi(R7, 4, R7);    //   sub R7, 2, R7   #   int x;
  stw(R0, R7, 1);
  movi(0, R2);        //   mov 0, R1
  stw(R2, R7, 0);     //   stw R2, R7[0]   #   x= 0;
  setLabel(loop);     // loop:             #   for (;;) {
  call(getchar);      //   call getchar    #     int c= getchar();
  ldw(R7, 0, R2);     //   ldw R7[0], R2   #
  bli(R1, '0', end);  //   bl R1, '0', end #     if (c>='0' &&
  bgi(R1, '9', end);  //   bg R1, '9', end #         c<='9') {
  muli(R2, 10, R2);   //   mul R2, 10, R2  #         x *= 10;
  add(R2, R1, R2);    //   add R2, R1, R2  #         x += c;
  subi(R2, '0', R2);  //   sub R2, '0', R2 #         x -= '0';
  stw(R2, R7, 0);     //   stw R2, R7[0]   #     }
  ba(loop);           //   ba loop         #     else break;
  setLabel(end);      // end:              #   }
  mov(R2, R1);        //   mov R2, R1      #   return x;
  ldw(R7, 1, R0);
  addi(R7, 4, R7);
  ret();              //   ret             # }

  getcharGen();

  readUintGenerated= 1;
  return readUint;
}

static int printIntGenerated= 0;

Label printIntGen() {
  Label printInt= getLabel("printInt");
  if (printIntGenerated)
    return printInt;

  Label printUint= getLabel("printUint");
  Label dash= getLabel("printInt$dash");
  Label printStr= getLabel("printStr");

  setLabel(printInt); //                   # void printInt(int x) { // R1
  bgei(R1, 0, printUint); //               #   if (x>=0)
                      //   bge R1, 0, printUint # return printUint(x);
  neg(R1, R1);        //   neg R1, R1      #   else {
  subi(R7, 4, R7);    //   sub R7, 4, R7);
  stw(R0, R7, 0);     //   stw R0, R7[0]
  stw(R1, R7, 1);     //   stw R1, R7[1]
  mova(dash, R1);     //   mov dash, R1
  call(printStr);     //   call printStr   #      printStr("-");
  ldw(R7, 0, R0);
  ldw(R7, 1, R1);
  addi(R7, 4, R7);
  ba(printUint);      //   ba printUint    #      return printUint(-x);
                      //                   # } }
  setLabel(dash);     // dash:
  string("-");        //   .string "-"

  printUintGen();
  printStrGen();

  printIntGenerated= 1;
  return printInt;
}

static int printUintGenerated= 0;

Label printUintGen() {
  Label printUint= getLabel("printUint");
  if (printUintGenerated)
    return printUint;

  Label loop1= getLabel("printInt$loop1");
  Label loop2= getLabel("printInt$loop2");

  setLabel(printUint);// printUint:         # void printUint(int x) { // R1
  movi(TERM, R4);     //   mov TERM, R4;   #   char *term= (char*)TERM;
  subi(R7, 6, R7);    //   sub R7, 2, R7   #   char digs[6];
  mov(R7, R2);        //   mov R7, R2      #   char *s= digs; // R2
  setLabel(loop1);    // loop1:            #   do {
  remi(R1, 10, R3);   //   rem R1, 10, R3  #     int c= x % 10;
  addi(R3, '0', R3);  //   add R3, '0', R3 #     c += '0';
  stb(R3, R2, 0);     //   stb R3, R2[0]   #     *s= c;
  addi(R2, 1, R2);    //   addi R2, 1, R2  #     s++;
  udivi(R1, 10, R1);  //   udiv R1, 10, R1 #     x /= 10;
  bnei(R1, 0, loop1); //   bne R1, 0, loop1#   } while (x!=0);
  setLabel(loop2);    // loop2:            #   do {
  subi(R2, 1, R2);    //   subi R2, 1, R2  #     s--;
  ldsb(R2, 0, R3);    //   ldsb R2[0], R3  #     char c= *s;
  stb(R3, R4, 0);     //   stb R3, R4[0]   #     *term= c;
  bge(R2, R7, loop2); //   bge R2,R7,loop2 #   } while (s>=digs);
  addi(R7, 6, R7);    //   addi(R7, 6, R7);
  ret();              //   ret             # }

  printIntGenerated= 1;
  return printUint;
}

static int printHexaGenerated= 0;

Label printHexaGen() {
  Label printHexa= getLabel("printHexa");
  if (printHexaGenerated)
    return printHexa;

  Label loop1= getLabel("printHexa$loop1");
  Label loop2= getLabel("printHexa$loop2");
  Label rango0_9= getLabel("printHexa$rango0_9");
  Label guardar= getLabel("printHexa$guardar");

  setLabel(printHexa); //printHexa:        # void printHexa(uint x) { // R1
  movi(TERM, R4);     //   mov TERM, R4;   #   char *term= (char*)TERM;
  subi(R7, 4, R7);    //   sub R7, 4, R7   #   char digs[4];
  mov(R7, R2);        //   mov R7, R2      #   char *s= digs; // R2
  setLabel(loop1);    // loop1:            #   do {
  andi(R1, 0xf, R3);  //   rem R1, 10, R3  #     int c= x & 0xf;
  bli(R3,10,rango0_9);//   bl R3,10,rango0_9#    if (c>=10)
  addi(R3, 'A'-10, R3);//  add R3, 'A', R3 #       c+= 'A'-10;
  ba(guardar);        //   ba guardar
  setLabel(rango0_9); // rango0_9:         #     else
  addi(R3, '0', R3);  //   add R3, '0', R3 #       c += '0';
  setLabel(guardar);  // guardar:
  stb(R3, R2, 0);     //   stb R3, R2[0]   #     *s= c;
  addi(R2, 1, R2);    //   addi R2, 1, R2  #     s++;
  shri(R1, 4, R1);    //   shr R1, 10, R1  #     x >>= 10;
  bnei(R1, 0, loop1); //   bne R1,0,loop1  #   } while (x!=0);
  setLabel(loop2);    // loop2:            #   do {
  subi(R2, 1, R2);    //   subi R2, 1, R2  #     s--;
  ldsb(R2, 0, R3);    //   ldsb R2[0], R3  #     char c= *s;
  stb(R3, R4, 0);     //   stb R3, R4[0]   #     *term= c;
  bge(R2, R7, loop2); //   bge R2,R7,loop2 #   } while (s>=digs);
  addi(R7, 4, R7);    //   addi(R7, 6, R7);
  ret();              //   ret             # }

  printHexaGenerated= 1;
  return printHexa;
}

// Instrucciones experimentales: son solo para propositos docentes

void ldwpp(Reg *ra, int idx, Reg *rd) {
  gen(opcode(OPldwpp) | regd(rd) | regs1(ra) | imm5(idx));
}

#ifdef OPswap
void swap(Reg *rs, Reg *ra, int idx) {
  gen(opcode(OPswap) | regd(rs) | regs1(ra) | imm5(idx));
}
#endif
