#include <nSystem.h>

#include "mutex.h"

// =================================================
// Implementacion de spin-locks
// =================================================

// No son verdaderos spinLocks, pero son funcionalmente equivalentes

void initSpinLock(SpinLock *psl, int ini) {
  psl->busy= ini;
  psl->m= nMakeMonitor();
}

void destroySpinLock(SpinLock *psl) {
  nDestroyMonitor(psl->m);
}

void spinLock(SpinLock *psl) {
  nEnter(psl->m);
  while (psl->busy==1)
    nWait(psl->m);
  psl->busy= 1;
  nExit(psl->m);
}

void spinUnlock(SpinLock *psl) {
  nEnter(psl->m);
  psl->busy= 0;
  nNotifyAll(psl->m);
  nExit(psl->m);
}

static int busy_ids[NCORES];

static __thread int core_id= -1;
static nMonitor mon;

void initCoreIds() {
  mon= nMakeMonitor();
}

void setCoreId() {
  nEnter(mon);
  int id= 0;
  while (id<NCORES) {
    if (busy_ids[id]==0)
      break;
    id++;
  }
  if (id<NCORES) {
    core_id= id;
    busy_ids[id]= TRUE;
  }
  else
    nFatalError("setCoreId", "Too much cores!\n");
  nExit(mon);
}

void freeCoreId() {
  nEnter(mon);
  busy_ids[core_id]= FALSE;
  nExit(mon);
}

int coreId() {
  return core_id;
}
    
// =================================================
// Implementacion de la cola de prioridades
// =================================================

#define MAXQSZ 1000

struct priqueue {
  void **vec;
  int *ofertas;
  int size, maxsize;
};

PriQueue MakePriQueue() {
  PriQueue pq= nMalloc(sizeof(*pq));
  pq->maxsize= MAXQSZ;
  pq->vec= nMalloc(sizeof(void*)*(pq->maxsize+1));
  pq->ofertas= nMalloc(sizeof(int)*(pq->maxsize+1));
  pq->size= 0;
  return pq;
}

void DestroyPriQueue(PriQueue pq) {
  if (pq->size!=0)
    nFatalError("DestroyPriQueue", "La cola de prioridad no esta vacia\n");
  nFree(pq->vec);
  nFree(pq->ofertas);
  nFree(pq);
}

void *PriGet(PriQueue pq) {
  void *t;
  int k;
  if (pq->size==0)
    return NULL;
  t= pq->vec[0];
  pq->size--;
  for (k= 0; k<pq->size; k++) {
    pq->vec[k]= pq->vec[k+1];
    pq->ofertas[k]= pq->ofertas[k+1];
  }
  return t;
}

void PriPut(PriQueue pq, void *t, int oferta) {
  if (pq->size==pq->maxsize)
    nFatalError("PriPut", "Desborde de la cola de prioridad\n");
  int k;
  for (k= pq->size-1; k>=0; k--) {
    if (oferta > pq->ofertas[k])
      break;
    else {
      pq->vec[k+1]= pq->vec[k];
      pq->ofertas[k+1]= pq->ofertas[k];
    }
  }
  pq->vec[k+1]= t;
  pq->ofertas[k+1]= oferta;
  pq->size++;
}

int PriSize(PriQueue pq) {
  return pq->size;
}

int PriBest(PriQueue pq) {
  return pq->size==0 ? 0 : pq->ofertas[0];
}

int EmptyPriQueue(PriQueue pq) {
  return pq->size==0;
}

int PriLength(PriQueue pq) {
  return pq->size;
}

