#define _DEFAULT_SOURCE 1

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>

#include "rifa.h"

typedef struct {
  Rifa *r;
  void *premio;
  int n, nro_concursos;
  int *pconcurso_actual;
  int *pcompletados;
  pthread_mutex_t *pm;
  pthread_cond_t *pc, *pc_comp;
  int ganados;
  pthread_t t;
} Param;

// ----------------------------------------------------
// Funcion que termina con un error si un valor es falso

static void verificar(int b, char *msg) {
  if (!b) {
    fprintf(stderr, "%s\n", msg);
    exit(1);
  }
}

static void *concursante(void *ptr) {
  Param *p= ptr;

  int concurso= 0;
  while (concurso<p->nro_concursos) {
    void *premio= concursar(p->r);
    if (premio!=NULL) {
      verificar(premio==p->premio, "El premio es incorrecto");
      p->ganados++;
    }
    
    concurso++;
    pthread_mutex_lock(p->pm);
    (*p->pcompletados)++;
    if (*p->pcompletados==p->n)
      pthread_cond_broadcast(p->pc_comp);
    while (concurso>*p->pconcurso_actual && concurso<p->nro_concursos)
      pthread_cond_wait(p->pc, p->pm);
    pthread_mutex_unlock(p->pm);
  }

  return NULL;
}

typedef struct {
  int n, nro_concursos, verificar, reportar;
} TestParam;

static void *test(void *ptr) {
  TestParam *tp= ptr;
  int n= tp->n;
  int nro_concursos= tp->nro_concursos;
  int *premios= malloc(nro_concursos);
  Param p[n];
  pthread_mutex_t m= PTHREAD_MUTEX_INITIALIZER;
  pthread_cond_t c= PTHREAD_COND_INITIALIZER;
  pthread_cond_t c_comp= PTHREAD_COND_INITIALIZER;
  int concurso_actual= 0;
  int completados= 0;
  Rifa *r= nuevaRifa(tp, n);

  for (int i= 0; i<n; i++) {
    p[i].r= r;
    p[i].premio= tp;
    p[i].n= n;
    p[i].nro_concursos= nro_concursos;
    p[i].pconcurso_actual= &concurso_actual;
    p[i].pcompletados= &completados;
    p[i].pm= &m;
    p[i].pc= &c;
    p[i].pc_comp= &c_comp;
    p[i].ganados= 0;
    if (pthread_create(&p[i].t, NULL, concursante, &p[i])) {
      perror("pthread_create");
      exit(1);
    }
  }

  while (concurso_actual<nro_concursos) {
    pthread_mutex_lock(&m);
    while (completados<n)
      pthread_cond_wait(&c_comp, &m);
    destruirRifa(r);
    concurso_actual++;
    completados= 0;
    if (concurso_actual<nro_concursos) {
      r= nuevaRifa(&premios[concurso_actual], n);
      for (int i= 0; i<n; i++) {
        p[i].r= r;
        p[i].premio= &premios[concurso_actual];
      }
    }
    pthread_cond_broadcast(&c);
    pthread_mutex_unlock(&m);
  }

  for (int i= 0; i<n; i++) {
    pthread_join(p[i].t, NULL);
  }

  free(premios);
  pthread_mutex_destroy(&m);
  pthread_cond_destroy(&c);
  pthread_cond_destroy(&c_comp);

  if (tp->reportar) {
    for (int i= 0; i<n; i++)
      printf("Concursante %d gano %d veces\n", i, p[i].ganados);
  }
  if (tp->verificar) { 
    int total= 0;
    double media= (double)nro_concursos/n;
    for (int i= 0; i<n; i++) {
      total += p[i].ganados;
      verificar(media*0.66<=p[i].ganados && p[i].ganados<media*1.5,
                "Uno de los threads no gana nro. de concursos esperado");
    }
    verificar(total==nro_concursos, "No se repartieron todos los premios");
  }

  return NULL;
}

int main() {
  printf("Test: 1 concursante, 1 sola rifa\n");
  {
    TestParam tp= {1, 1, 1, 1};
    test(&tp);
    printf("Test aprobado\n");
  }

  printf("=================================\n");
  printf("Test: 2 concursantes, 1 sola rifa\n");
  {
    TestParam tp= {2, 1, 0, 1};
    test(&tp);
    printf("Test aprobado\n");
  }

  printf("=============================\n");
  printf("Test: 2 concursantes, 5 rifas\n");
  {
    TestParam tp= {5, 5, 0, 1};
    test(&tp);
    printf("Test aprobado\n");
  }

  printf("=================================\n");
  {
#ifdef VALGRIND
    TestParam tp= {10, 10000, 1, 1};
#else
    TestParam tp= {10, 25000, 1, 1};
#endif
    printf("Test: 10 concursantes, %d rifas\n", tp.nro_concursos);
    test(&tp);
    printf("Test aprobado\n");
  }

  printf("=================================\n");
  printf("Test: multiples rifas en paralelo\n");
  printf("(solo se reportan los resultados de la primera rifa)\n");
  {
#ifdef VALGRIND
    int q= 7;
    int nro_concursos= 2000;
#else
    int q= 10;
    int nro_concursos= 10000;
#endif
    TestParam tp[q];
    pthread_t t[q];
    for (int i= 0; i<q; i++) {
      tp[i].n= q/2+i/2;
      tp[i].nro_concursos= nro_concursos/2+nro_concursos/(i+1);
      printf("%d participantes, %d rifas\n", tp[i].n, tp[i].nro_concursos);
      tp[i].verificar= 1;
      tp[i].reportar= i==0;
      pthread_create(&t[i], NULL, test, &tp[i]);
    }
    for (int i= 0; i<q; i++) {
      pthread_join(t[i], NULL);
    }
    printf("Test aprobado\n");
  }

  printf("\nFelicitaciones: paso todos los tests\n");

  return 0;
}
