#define _DEFAULT_SOURCE 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "menores.h"

// ----------------------------------------------------
// Version secuencial de la funcion que selecciona los menores.
// Retorna la cantidad k de elementos del arreglo a de tamano n que son
// menores que x.  Todos los elementos que son menores que x quedan
// almacenados en a[0], ..., a[k-1], respetando el orden que tenian en
// el arreglo original

int menores(double a[], int n, double x) {
  int k= 0;
  for (int i= 0; i<n; i++) {
    if (a[i]<x)
      a[k++]= a[i];
  }
  return k;
}

// ----------------------------------------------------
// La funcion leer para recibir n bytes de un pipe

int leer(int fd, void *vbuf, int n) {
  char *buf= vbuf;
  do {
    int rc= read(fd, buf, n);
    if (rc<=0)
      return 1; /* fracaso: error o fin del archivo/pipe/socket */
    n-= rc; /* descontamos los bytes leídos */
    buf+= rc; /* avanzamos el buffer para no reescribir lo leido previamente */
  } while (n>0); /* mientras no leamos todo lo que esperamos */
  return 0; /* exito */
}

// ----------------------------------------------------
// Funcion que entrega el tiempo transcurrido desde el lanzamiento del
// programa en milisegundos

static long long time0= 0;

static long long getTime0() {
    struct rusage ru;
    int rc= getrusage(RUSAGE_SELF, &ru);
    if (rc!=0) {
      perror("getTime");
      exit(1);
    }
    return (long long)ru.ru_utime.tv_sec*1000000+ru.ru_utime.tv_usec;
}

static void resetTime() {
  time0= getTime0();
}

static int getTime() {
  return (getTime0()-time0+500)/1000;
}

// ----------------------------------------------------
// Funcion que termina con un error si un valor es falso

static void verificar(int b, char *msg) {
  if (!b) {
    fprintf(stderr, "%s\n", msg);
    exit(1);
  }
}

int main() {
  printf("Test: arreglo ascendente con 2 elementos\n");

  {
    double a[]= {2.71, 3.14};
    int k= menoresPar(a, 2, 3.0);
    verificar(k==1, "Valor retornado no es 1");
    verificar(a[0]==2.71, "Primer elemento no es 2.71");
    printf("Test aprobado\n");
  }

  printf("Test: arreglo descendente con 2 elementos\n");

  {
    double a[]= {3.14, 2.71};
    int k= menoresPar(a, 2, 3.0);
    verificar(k==1, "Valor retornado no es 1");
    verificar(a[0]==2.71, "Primer elemento no es 2.71");
    printf("Test aprobado\n");
  }

  printf("Test: arreglo de 10 elementos\n");

  {
    double a[]= {5, 1, 7, 8, 4, 2, 6, 10, 9, 3};
    for (int kesp= 0; kesp<=10; kesp++) {
      double b[10];
      memcpy(b, a, 10*sizeof(double));
      int k= menoresPar(b, 10, kesp+1);
      verificar(k==kesp, "Valor retornado es incorrecto");
      int j= 0;
      for (int i=0; i<10; i++) {
        if (a[i]<kesp+1) {
          verificar(a[i]==b[j], "Elemento es incorrecto");
          j++;
        }
      }
    }
    printf("Test aprobado\n");
  }

#ifdef VALGRIND
  int N= 5000000; // Numero de elementos del arreglo
  double X= (double)RAND_MAX/10000; // Se seleccionan los menores que X
  double META= 1.0;  // Speed up requerido
#else
  int N= 50000000; // Numero de elementos del arreglo
  double X= (double)RAND_MAX/10000; // Se seleccionan los menores que X
  double META= 1.5;  // Speed up requerido
#endif

  printf("=============================================\n");
  printf("Benchmark: arreglo de %d elementos aleatorios\n", N);
  double *a= malloc(N*sizeof(double));
  double *b= malloc(N*sizeof(double));
  for (int i= 0; i<N; i++)
    a[i]= (double)random()/100;

  int intento;
  for (intento= 1; intento<=5; intento++) {

    printf("Intento %d\n", intento);
    printf("Precalentamiento ejecucion secuencial\n");
    memcpy(b, a, N*sizeof(double));
    int k_sec= menores(b, N, X);
    printf("Medicion tiempo de ejecucion de la ejecucion secuencial\n");
    memcpy(b, a, N*sizeof(double));
    resetTime();
    k_sec= menores(b, N, X);
    int tiempo_sec= getTime();
    printf("Tiempo= %d ms.  %d elementos seleccionados.  Los primeros 5 son:\n",
           tiempo_sec, k_sec);
    for (int i= 0; i<5;i++)
      printf("%f ", a[i]);
    printf("\n");
    int j= 0;
    for (int i=0; i<N; i++) {
      if (a[i]<X) {
        verificar(a[i]==b[j],
                  "Version secuencial selecciona valor incorrecto!");
        j++;
      }
    }
    verificar(j==k_sec, "Version secuencial retorna resultado incorrecto");

    printf("Precalentamiento ejecucion **paralela**\n");
    memcpy(b, a, N*sizeof(double));
    int k_par= menoresPar(b, N, X);
    printf("Medicion tiempo de ejecucion de la ejecucion **paralela**\n");
    memcpy(b, a, N*sizeof(double));
    resetTime();
    k_par= menoresPar(b, N, X);
    int tiempo_par= getTime();
    printf("Tiempo= %d ms.  %d elementos seleccionados.  Los primeros 5 son:\n",
           tiempo_par, k_sec);
    for (int i= 0; i<5;i++)
      printf("%f ", a[i]);
    printf("\n");
    j= 0;
    for (int i=0; i<N; i++) {
      if (a[i]<X) {
        verificar(a[i]==b[j],
                  "Version paralela selecciona valor incorrecto!");
        j++;
      }
    }
    verificar(j==k_par, "Version paralela retorna resultado incorrecto");
    verificar(k_sec==k_par, "No se retorna mismo valor que version secuencial");
    double speedup= (double)tiempo_sec/tiempo_par;
    printf("Speed up= %f\n", speedup);
    if (speedup>=META)
      break;
    else {
      printf("El speed up es inferior a %f\n", META);
      if (intento<5)
        printf("Se hara un nuevo intento\n");
      printf("------------------------------\n");
    }
  }
  free(a);
  free(b);
  if (intento>5) {
    printf("No se alcanza speed up deseado despues de 5 intentos\n");
    printf("Podria mejorar, pero la pregunta no lo exige\n");
  }

  printf("Test aprobado\n");

  printf("Felicitaciones: paso todos los tests\n");

  return 0;
}
