#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdint.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "arboles.h"

// ----------------------------------------------------
// Funcion que entrega el tiempo transcurrido desde 
// la llamada a resetTime()

static long long time0= 0;

static long long getTime0() {
    struct rusage ru;
    int rc= getrusage(RUSAGE_SELF, &ru);
    if (rc!=0) {
      perror("getTime");
      exit(1);
    }
    return (long long)ru.ru_utime.tv_sec*1000000+ru.ru_utime.tv_usec;
}

static void resetTime() {
  time0= getTime0();
}

static int getTime() {
  return (getTime0()-time0+500)/1000;
}

// ----------------------------------------------------
// Funcion que construye un arbol binario a partir de otros 2

static Nodo *abb(int x, Nodo *izq, Nodo *der) {
  Nodo *a= malloc(sizeof(*a));
  a->x= x;
  a->izq= izq;
  a->der= der;
  return a;
}

// ----------------------------------------------------
// Funcion que construye un arbol consistente en una sola hoja

static Nodo *hoja(int x) {
  return abb(x, NULL, NULL);
}

// ----------------------------------------------------
// Crea un arbol binario de busqueda bien equilibrado en donde 
// los x van de i a j

static Nodo *equilibrado(int i, int j) {
  if (i>j)
    return NULL;
  int k= (i+j+1)/2;
  Nodo *a= malloc(sizeof(Nodo));
  a->x= k;
  a->izq= equilibrado(i, k-1);
  a->der= equilibrado(k+1, j);
  return a;
}

// ----------------------------------------------------
// Funcion que termina con un error si un valor es falso

static void verificar(int b, char *msg) {
  if (!b) {
    fprintf(stderr, "%s\n", msg);
    exit(1);
  }
}

// ----------------------------------------------------
// Verifica que un lista doblemente enlazada contiene
// los elementos de un arreglo

static void verificar_lista(Nodo *a, int elems[], int n) {
  Nodo *prev= NULL;
  for (int i= 0; i<n; i++) {
    verificar(a->izq==prev, "El nodo previo incorrecto");
    verificar(a->x == elems[i], "La etiqueta del nodo es incorrecta");
    prev= a;
    a= a->der;
  }
  verificar(a==NULL, "El nodo siguiente del ultimo nodo no es nulo");
}

// ----------------------------------------------------
// Libera una lista doblemente enlazada

static void liberar_lista(Nodo *a) {
  while (a!=NULL) {
    Nodo *prev= a;
    a= a->der;
    free(prev);
  }
}

int main() {
  printf("Un arbol de una sola hoja\n");

  {
    Nodo *a= hoja(5);
    int elems[]= {5};
    Nodo *prim, *ult;
    abb2lde(a, &prim, &ult);
    verificar(prim==a, "El primer nodo es incorrecto");
    verificar(ult==a, "El ultimo nodo es incorrecto");
    verificar_lista(prim, elems, 1);
    liberar_lista(prim);
    printf("Test aprobado\n");
  }

  printf("Arbol con un nodo con un hijo a la derecha\n");

  {
    Nodo *h= hoja(5);
    Nodo *a= abb(2, NULL, h);
    int elems[]= {2, 5};
    Nodo *prim, *ult;
    abb2lde(a, &prim, &ult);
    verificar(prim==a, "El primer nodo es incorrecto");
    verificar(ult==h, "El ultimo nodo es incorrecto");
    verificar_lista(prim, elems, 2);
    liberar_lista(prim);
    printf("Test aprobado\n");
  }

  printf("Arbol con un nodo con un hijo a la izquierda\n");

  {
    Nodo *h= hoja(2);
    Nodo *a= abb(5, h, NULL);
    int elems[]= {2, 5};
    Nodo *prim, *ult;
    abb2lde(a, &prim, &ult);
    verificar(prim==h, "El primer nodo es incorrecto");
    verificar(ult==a, "El ultimo nodo es incorrecto");
    verificar_lista(prim, elems, 2);
    liberar_lista(prim);
    printf("Test aprobado\n");
  }

  printf("Arbol con un nodo con sus 2 hijos\n");

  {
    Nodo *izq= hoja(2);
    Nodo *der= hoja(7);
    Nodo *a= abb(5, izq, der);
    int elems[]= {2, 5, 7};
    Nodo *prim, *ult;
    abb2lde(a, &prim, &ult);
    verificar(prim==izq, "El primer nodo es incorrecto");
    verificar(ult==der, "El ultimo nodo es incorrecto");
    verificar_lista(prim, elems, 3);
    liberar_lista(prim);
    printf("Test aprobado\n");
  }
  
  printf("El ejemplo del enunciado\n");

  {
    Nodo *izq= abb(2, NULL, hoja(3));
    Nodo *der= abb(7, hoja(6), NULL);
    Nodo *a= abb(5, izq, der);
    int elems[]= {2, 3, 5, 6, 7};
    Nodo *prim, *ult;
    abb2lde(a, &prim, &ult);
    verificar(prim==izq, "El primer nodo es incorrecto");
    verificar(ult==der, "El ultimo nodo es incorrecto");
    verificar_lista(prim, elems, 5);
    liberar_lista(prim);
    printf("Test aprobado\n");
  }

  printf("Verificando linealidad del tiempo de ejecucion\n");

#ifdef VALGRIND
#define N 100000
#else
#define N 10000000
#endif

  int k= 0;
  for (k= 1; k<=5; k++) {
    printf("Intento %d\n", k);
    int *elems= malloc(2*N*sizeof(int));
    for (int i= 0; i<2*N; i++)
      elems[i]= i;
    Nodo *prim, *ult;

    Nodo *a= equilibrado(0, N-1);
    resetTime();
    abb2lde(a, &prim, &ult);
    int tiempo_a= getTime();
    printf("Tiempo para convertir arbol de %d elementos = %d milisegundos\n",
           N, tiempo_a);
    verificar_lista(prim, elems, N);
    liberar_lista(prim);

    Nodo *b= equilibrado(0, N*2-1);
    resetTime();
    abb2lde(b, &prim, &ult);
    int tiempo_b= getTime();
    printf("Tiempo para convertir arbol de %d elementos = %d milisegundos\n",
           2*N, tiempo_b);
    verificar_lista(prim, elems, 2*N);
    liberar_lista(prim);
    free(elems);
    int tolerancia= 3*tiempo_a;
    printf("Maximo razonable para el segundo tiempo = %d\n", tolerancia);
#ifdef VALGRIND
    break;
#else
    if (tiempo_b>tolerancia)
      printf("Segundo tiempo excede 3 veces el primer tiempo\n");
    else
      break;
#endif
  }
  if (k>5) {
    fprintf(stderr, "Tiempo no es lineal\n");
    exit(1);
  }
  else
    printf("Test aprobado\n");

  printf("Felicitaciones: paso todos los tests\n");

  return 0;
}
