/*
 * main.c -- the bare jpipe char module
Cuando mato el open, queda rdrs=1
 *
 * Copyright (C) 2001 Alessandro Rubini and Jonathan Corbet
 * Copyright (C) 2001 O'Reilly & Associates
 * Original: scull driver
 * Modified: Jo Piquer, 2008
 *
 * The source code in this file can be freely used, adapted,
 * and redistributed in source or binary form, so long as an
 * acknowledgment appears in derived source files.  The citation
 * should list that the code comes from the book "Linux Device
 * Drivers" by Alessandro Rubini and Jonathan Corbet, published
 * by O'Reilly & Associates.   No warranty is attached;
 * we cannot take responsibility for errors or fitness for use.
 *
 */

#include <linux/autoconf.h> /* JO */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>	/* O_ACCMODE */
#include <linux/seq_file.h>
#include <linux/cdev.h>
#include <linux/sched.h>

#include <asm/system.h>		/* cli(), *_flags */
#include <asm/uaccess.h>	/* copy_*_user */
#include <asm/current.h>

#include "jpipe.h"		/* local definitions */

/*
 * Our parameters which can be set at load time.
 */

int jpipe_major =   JPIPE_MAJOR;
int jpipe_minor =   0;
int jpipe_nr_devs = JPIPE_NR_DEVS;	/* number of bare jpipe devices */
int jpipe_size = JPIPE_MAX_BUF;

module_param(jpipe_major, int, S_IRUGO);
module_param(jpipe_minor, int, S_IRUGO);
module_param(jpipe_nr_devs, int, S_IRUGO);
module_param(jpipe_size, int, S_IRUGO);

MODULE_AUTHOR("Alessandro Rubini, Jonathan Corbet, Jo Piquer");
MODULE_LICENSE("Dual BSD/GPL");

struct jpipe_dev *jpipe_devices;	/* allocated in jpipe_init_module */


#ifdef JPIPE_DEBUG /* use proc only if debugging */
/*
 * The proc filesystem: function to read and entry
 */

int jpipe_read_procmem(char *buf, char **start, off_t offset,
                   int count, int *eof, void *data)
{
    int i, len = 0;
    int limit = count - 80; /* Don't print more than this */

    for (i = 0; i < jpipe_nr_devs && len <= limit; i++) {
	struct jpipe_dev *d = &jpipe_devices[i];
	if (down_interruptible(&d->sem))
	    return -ERESTARTSYS;
	if(!(d->status&JPIPE_USED))
	    len += sprintf(buf+len,"\nDevice %i: not used\n", i);
	else
	    len += sprintf(buf+len,"\nDevice %i: size=%li, bytes=%li, rdrs=%li, wrtrs=%li\n", i, (long)d->size, (long)d->nbytes, (long)d->nrdr, (long)d->nwrt);
	up(&d->sem);
   }
   *eof = 1;
   return len;
}

/*
 * Actually create (and remove) the /proc file(s).
 */

static void jpipe_create_proc(void)
{
    create_proc_read_entry("jpipestatus", 0 /* default mode */,
			                 NULL /* parent dir */,
                                         jpipe_read_procmem,
			                 NULL /* client data */);
}

static void jpipe_remove_proc(void)
{
    /* no problem if it was not registered */
    remove_proc_entry("jpipestatus", NULL /* parent dir */);
}


#endif /* JPIPE_DEBUG */

/*
 * Open and close: Sync on open, wait until at least 1 reader and 1 writer
 *                               are present
 */

int jpipe_open(struct inode *inode, struct file *filp)
{
    struct jpipe_dev *dev; /* device information */

    dev = container_of(inode->i_cdev, struct jpipe_dev, cdev);
    filp->private_data = dev; /* for other methods */

    if (down_interruptible(&dev->sem))
	return -ERESTARTSYS;

    if(!(dev->status&JPIPE_USED)) {   /* first open */
	dev->status = JPIPE_USED;
	dev->buf = kmalloc(jpipe_size, GFP_KERNEL);
        if (!dev->buf) {
           up(&dev->sem);
           return -ENOMEM;
        }
	dev->in = dev->out = dev->buf;
	dev->end = dev->buf + jpipe_size;
	dev->nbytes = 0;
	dev->size = jpipe_size;
	dev->nrdr = dev->nwrt = 0;
    }

    if(filp->f_mode & FMODE_READ)
	dev->nrdr++;
    if(filp->f_mode & FMODE_WRITE)
	dev->nwrt++;

    /* wait for readers and writers */
    while(dev->nrdr == 0 || dev->nwrt == 0) {
	up(&dev->sem); /* release the lock */
        if (filp->f_flags & O_NONBLOCK) {
    	    if(filp->f_mode & FMODE_READ)
		dev->nrdr--;
     	    if(filp->f_mode & FMODE_WRITE)
		dev->nwrt--;

    	    if(dev->nrdr == 0 && dev->nwrt == 0) {
		kfree(dev->buf);
		dev->buf = NULL;
		dev->status = JPIPE_NOTUSED;
	    }

            return -EAGAIN;
	}
        PDEBUG("\"%s\" opening: going to sleep\n", current->comm);
        if (wait_event_interruptible(dev->r_queue,
				     (dev->nrdr != 0 && dev->nwrt != 0))) {
    	    if(filp->f_mode & FMODE_READ)
		dev->nrdr--;
     	    if(filp->f_mode & FMODE_WRITE)
		dev->nwrt--;

    	    if(dev->nrdr == 0 && dev->nwrt == 0) {
		kfree(dev->buf);
		dev->buf = NULL;
		dev->status = JPIPE_NOTUSED;
	    }
            return -ERESTARTSYS; /* signal: tell the fs layer to handle it */
	}
        /* otherwise loop, but first reacquire the lock */
        if (down_interruptible(&dev->sem)) {
    	    if(filp->f_mode & FMODE_READ)
		dev->nrdr--;
     	    if(filp->f_mode & FMODE_WRITE)
		dev->nwrt--;

    	    if(dev->nrdr == 0 && dev->nwrt == 0) {
		kfree(dev->buf);
		dev->buf = NULL;
		dev->status = JPIPE_NOTUSED;
	    }
            return -ERESTARTSYS; /* signal: tell the fs layer to handle it */
	}
    }

    wake_up_interruptible(&dev->r_queue);

    up(&dev->sem);
    PDEBUG("\"%s\" opening: OK\n", current->comm);

    return nonseekable_open(inode, filp);
}

int jpipe_release(struct inode *inode, struct file *filp)
{
    struct jpipe_dev *dev = filp->private_data; 

    down(&dev->sem); /* Non-interruptable: beware! */

    if(filp->f_mode & FMODE_READ)
	dev->nrdr--;
    if(filp->f_mode & FMODE_WRITE)
	dev->nwrt--;

    if(dev->nrdr == 0)
	dev->status |= JPIPE_CLOSED;
    if(dev->nwrt == 0)
	dev->status |= JPIPE_CLOSED;

    if(dev->nrdr == 0 && dev->nwrt == 0) {
	kfree(dev->buf);
	dev->buf = NULL;
	dev->status = JPIPE_NOTUSED;
    }

    /* finally, awake any writers and readers */
    wake_up_interruptible_all(&dev->r_queue);
    wake_up_interruptible_all(&dev->w_queue);
    up(&dev->sem);

    PDEBUG("\"%s\" release device\n", current->comm);
    return 0;
}
 
/*
 * Read and write
 */

ssize_t jpipe_read(struct file *filp, char __user *buf, size_t count,
                   loff_t *f_pos)
{
    struct jpipe_dev *dev = filp->private_data; 

    if (down_interruptible(&dev->sem))
	return -ERESTARTSYS;

    while (dev->nbytes == 0 && !(dev->status&JPIPE_CLOSED)) { /* nothing to read */
        up(&dev->sem); /* release the lock */
      /* Soporte para NONBLOCK */
        if (filp->f_flags & O_NONBLOCK)
            return -EAGAIN;

        PDEBUG("\"%s\" reading: going to sleep\n", current->comm);

        if (wait_event_interruptible(dev->r_queue,
                              (dev->nbytes > 0 || (dev->status&JPIPE_CLOSED)) ))
            return -ERESTARTSYS; /* signal: tell the fs layer to handle it */
            /* otherwise loop, but first reacquire the lock */
            if (down_interruptible(&dev->sem))
                return -ERESTARTSYS;
    }

    if(dev->nbytes == 0 && (dev->status&JPIPE_CLOSED)) {
	up(&dev->sem);
	return 0;
    }

    /* ok, data is there, return something */
    count = (count < dev->nbytes) ? count : dev->nbytes;
    if (dev->in < dev->out) /* wrap around */
	count = (count <= (dev->end-dev->out)) ? count : dev->end-dev->out;
    if (copy_to_user(buf, dev->out, count)) {
        up (&dev->sem);
        return -EFAULT;
    }
    
    dev->out += count;
    dev->nbytes -= count;
    if(dev->out == dev->end) dev->out = dev->buf; /* wrapped */

    up (&dev->sem);

    /* finally, awake any writers and return */
    wake_up_interruptible(&dev->w_queue);
    PDEBUG("\"%s\" did read %li bytes from pipe\n",current->comm, (long)count);
    return count;
}

ssize_t jpipe_write(struct file *filp, const char __user *buf, size_t count,
                    loff_t *f_pos)
{
    struct jpipe_dev *dev = filp->private_data;
    int written  = 0; /* bytes already written */
    int cnt;

    if (down_interruptible(&dev->sem))
	return -ERESTARTSYS;

retry:
    while (dev->nbytes == dev->size && count > written && !(dev->status&JPIPE_CLOSED)) { /* impossible to write */
        up(&dev->sem); /* release the lock */
        /* Soporte para NONBLOCK */
        if (filp->f_flags & O_NONBLOCK) {
	    if(written == 0)
                return -EAGAIN;
	    else
		return written;
	}

        PDEBUG("\"%s\" writing: going to sleep\n", current->comm);
        if (wait_event_interruptible(dev->w_queue,
                       (dev->nbytes < dev->size || (dev->status&JPIPE_CLOSED))))
            return -ERESTARTSYS; /* signal: tell the fs layer to handle it */
        /* otherwise loop, but first reacquire the lock */
        if (down_interruptible(&dev->sem))
            return -ERESTARTSYS;
    }

    if(dev->status&JPIPE_CLOSED) {
	up(&dev->sem);
	send_sig(SIGPIPE,current,0);
	return -EPIPE;
    }


    PDEBUG("\"%s\" writing: got space\n", current->comm);
    /* ok, some space is available, use it */
    cnt = count-written;
    cnt = (cnt < dev->size-dev->nbytes) ? cnt : dev->size-dev->nbytes;
    if (dev->out < dev->in) /* wrap around */
	cnt = (cnt < (dev->end-dev->in)) ? cnt : dev->end-dev->in;
    if (copy_from_user(dev->in, buf, cnt)) {
        up (&dev->sem);
        return -EFAULT;
    }
    
    dev->in += cnt;
    dev->nbytes += cnt;
    if(dev->in == dev->end) dev->in = dev->buf; /* wrapped */
    written += cnt;
    buf += cnt;
    /* awake any readers and continue */
    wake_up_interruptible(&dev->r_queue);

    if(written < count) goto retry;
    
    up (&dev->sem);

    PDEBUG("\"%s\" did write %li bytes to pipe\n",current->comm, (long) count);
    return count;
}

/*
 * The ioctl() implementation
 */

int jpipe_ioctl(struct inode *inode, struct file *filp,
                 unsigned int cmd, unsigned long arg)
{

/* Nothing yet */
	return 0;
}

struct file_operations jpipe_fops = {
	.owner =    THIS_MODULE,
	.llseek =   no_llseek,
	.read =     jpipe_read,
	.write =    jpipe_write,
	.ioctl =    jpipe_ioctl,
	.open =     jpipe_open,
	.release =  jpipe_release,
};

/*
 * Finally, the module stuff
 */

/*
 * The cleanup function is used to handle initialization failures as well.
 * Thefore, it must be careful to work correctly even if some of the items
 * have not been initialized
 */
void jpipe_cleanup_module(void)
{
    int i;
    dev_t devno = MKDEV(jpipe_major, jpipe_minor);

    /* Get rid of our char dev entries */
    if (jpipe_devices) {
	for (i = 0; i < jpipe_nr_devs; i++) {
	    cdev_del(&jpipe_devices[i].cdev);
	}
	kfree(jpipe_devices);
    }

#ifdef JPIPE_DEBUG /* use proc only if debugging */
    jpipe_remove_proc();
#endif

    /* cleanup_module is never called if registering failed */
    unregister_chrdev_region(devno, jpipe_nr_devs);
}


/*
 * Set up the char_dev structure for this device.
 */
static void jpipe_setup_cdev(struct jpipe_dev *dev, int index)
{
    int err, devno = MKDEV(jpipe_major, jpipe_minor + index);
    
    cdev_init(&dev->cdev, &jpipe_fops);
    dev->cdev.owner = THIS_MODULE;
    dev->cdev.ops = &jpipe_fops;
    err = cdev_add (&dev->cdev, devno, 1);
    /* Fail gracefully if need be */
    if (err)
	printk(KERN_NOTICE "Error %d adding jpipe%d", err, index);
}


int jpipe_init_module(void)
{
    int result, i;
    dev_t dev = 0;

/*
 * Get a range of minor numbers to work with, asking for a dynamic
 * major unless directed otherwise at load time.
 */
    if (jpipe_major) {
	dev = MKDEV(jpipe_major, jpipe_minor);
	result = register_chrdev_region(dev, jpipe_nr_devs, "jpipe");
    } else {
	result = alloc_chrdev_region(&dev, jpipe_minor, jpipe_nr_devs,
			             "jpipe");
	jpipe_major = MAJOR(dev);
    }
    if (result < 0) {
	printk(KERN_WARNING "jpipe: can't get major %d\n", jpipe_major);
	return result;
    }

    /* 
     * allocate the devices -- we can't have them static, as the number
     * can be specified at load time
     */
    jpipe_devices = kmalloc(jpipe_nr_devs * sizeof(struct jpipe_dev), GFP_KERNEL);
    if (!jpipe_devices) {
	result = -ENOMEM;
	goto fail;  /* Make this more graceful */
    }
    memset(jpipe_devices, 0, jpipe_nr_devs * sizeof(struct jpipe_dev));

    /* Initialize each device. */
    for (i = 0; i < jpipe_nr_devs; i++) {
	jpipe_devices[i].status = JPIPE_NOTUSED;
	init_waitqueue_head(&(jpipe_devices[i].r_queue));
	init_waitqueue_head(&(jpipe_devices[i].w_queue));
	init_MUTEX(&jpipe_devices[i].sem);
	jpipe_setup_cdev(&jpipe_devices[i], i);
    }

#ifdef JPIPE_DEBUG /* only when debugging */
    jpipe_create_proc();
#endif

    PDEBUG("jpipe module loaded\n");
    return 0; /* succeed */

fail:
    jpipe_cleanup_module();
    return result;
}

module_init(jpipe_init_module);
module_exit(jpipe_cleanup_module);
