
/*

Nrgrep -- a fast and flexible pattern matching tool.
Copyright (C) 2000 Gonzalo Navarro

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Author's contact: Gonzalo Navarro, Dept. of Computer Science, University of 
Chile. Blanco Encalada 2120, Santiago, Chile. gnavarro@dcc.uchile.cl

*/

	/* Search a regular expression in a text allowing errors */

#include "eregular.h"
#include "esimple.h"
#include "eextended.h"
#include "parser.h"

#define BK 0.78  /* k+1 has to be at most this to prefer it over fwd/bwd */

static double findBestMulti (Tree *e, Mask **trans, int m, int s, int *chMap,
                        Mask *B, Mask initial, Mask final, int *wlen,
                        Mask *dactive, Mask *dinitial, Mask *dfinal,
                        Mask *factive, Mask *finitial, Mask *ffinal,
                        Mask *bactive, Mask *binitial, Mask *bfinal, int K)

        /* Finds K+1 best subexpressions to search and puts their states in
	   dactive,dinitial,dfinal and in wlen the window length (0 if it 
	   recommends forward scanning). It also puts in factive,finitial,ffinal
	   and bactive,binitial,bfinal the states for fwd and bwd verification.
           Returns the avg number of chars inspected. All those masks arrays
	   are allocated, but their contents are not */

   { double *prob;    /* prob[i] = prob of arriving to state i from prev */
     double *pprob;   /* pprob[i,l] = prob matching a path of length l from
                                     state i */
     Mask *initials;  /* initial states at distance i from start state */
     double *mprob;   /* mprob[i,l,l'] = prob of any factor of length l'
                         inside a path of length l departing from state i */
     double *pcost;    /* pcost[i,l] = avg cost per window in same area */
     double *mbest; /* mbest[i,k] = cost using best selection of k
                       strings starting at initials[i]. the length of the 
		       strings is fixed */
     int *ibest;    /* where to start the first strings to obtain the
                       mbest cost */
     Mask reach,tmp,*ralltrans;
     int k,i,c,j,wl,l,lp,L,L2;
     int tr = OptTransp ? 1 : 0;
     double this,best,cost;

                /* compute max wlen wl <= m, in O(wl*m^2/w) = 
		   O(m^3/w) time */

     s--;
     reach = createMask(m);
     tmp = createMask(m);
     wl = 1; /* there is an extra initial state */
     SET(ZERO(reach,m),0);
     while (ISZERO(AND(COPY(tmp,reach,m),final,m),m))
        { wl++;
          ZERO(tmp,m);
          for (j=0;j<m;j++)
             if (ISSET(reach,j)) OR (tmp,trans[j][s],m);
                OR(reach,tmp,m);
        }
     L = 1+min(wl-1-tr*K,W)/(K+1);  /* 1+max length of pieces */
     L2 = L*L;

		/* compute initial states, O(wl*m^2/w) time */

     initials = malloc (wl * sizeof(Mask));
     initials[0] = createMasks (wl+1,m);
     for (i=1;i<wl;i++) initials[i] = initials[0] + i*maskSize(m);
     COPY(initials[0],initial,m); COPY(reach,initial,m);
     for (i=1;i<wl;i++)
        { ZERO(tmp,m);
          for (j=0;j<m;j++)
              if (ISSET(reach,j)) OR(tmp,trans[j][s],m);
	  AND(NOT(COPY(initials[i],reach,m),m),tmp,m);
	  OR(reach,tmp,m);
        }
     free (tmp); free (reach);
     
                /* load letter probabilities in O(m*s) time */

     prob = malloc (m * sizeof (double));
     for (i=0;i<m;i++)
         { prob[i] = 0.0;
           for (c=0;c<256;c++)
               if (ISSET(B[c],i)) prob[i] += letterProb[c];
         }

                /* compute path probabilities in O(wl*m^2*s/k) time */
                /* pprob[i,l] includes already the prob. of reaching i
                   from its predecessor */

     pprob = malloc (m * L * sizeof (double));
     for (i=0;i<m;i++)
        { pprob[i*L+0] = 1.0;
          pprob[i*L+1] = prob[i];
        }
     for (l=2;l<L;l++)
        for (i=0;i<m;i++)
           { pprob[i*L+l] = 0.0;
             if (!ISZERO(trans[i][s],m))
                for (j=0;j<m;j++)
                   if (ISSET(trans[i][s],j))
                      pprob[i*L+l] += pprob[j*L+(l-1)];
             pprob[i*L+l] *= prob[i];
	     if (pprob[i*L+l] > 1.0) pprob[i*L+l] = 1.0;
           }
     free (prob);

                /* compute mprob in O((m*wl/k)^2) time */

     mprob = malloc (m * L2 * sizeof (double));
     for (i=0;i<m;i++)
        for (l=0;l<L;l++)
           mprob[i*L2+l*L+0] = 1.0;
     for (l=1;l<L;l++)
        for (lp=1;lp<=l;lp++)
           for (i=0;i<m;i++)
              { mprob[i*L2+l*L+lp] = pprob[i*L+lp];
                if (lp < l)
                   if (!ISZERO(trans[i][s],m))
                      for (j=0;j<m;j++)
                         if (ISSET(trans[i][s],j))
			    mprob[i*L2+l*L+lp] = 1-(1-mprob[i*L2+l*L+lp])*
                                		   (1-mprob[j*L2+(l-1)*L+lp]);
              }
     free (pprob);

                /* compute pcost in O(m*(wl/k)^2) time */

     pcost = malloc (wl * L * sizeof (double));
     for (i=0;i<wl;i++)
        for (l=0;l<L;l++)
           { double new = 1.0;
	     for (j=0;j<m;j++)
		if (ISSET(initials[i],j))
                   for (lp=1;lp<=l;lp++) 
		      new += mprob[j*L2+l*L+lp];
             pcost[i*L+l] = new;
           }
     free (mprob);

                /* find best factors. this takes O(wl^2) time in the worst 
		   case but the average should be closer to O(k*wl) */

     mbest = malloc ((wl+1) * (K+2) * sizeof (double));
     ibest = malloc ((wl+1) * (K+2) * sizeof (int));
     dinitial[0] = createMasks (K+1,m);
     for (k=1;k<=K;k++) dinitial[k] = dinitial[0] + k*maskSize(m);
     *wlen = 0; best = BK;
     for (l = L-1; l > 1; l--)
        { if (best < 1/(double)l) break;  /* cannot win */
          for (i=1;i<=wl;i++) mbest[i*(K+2)+(0)] = 0.0;
          for (k=1;k<=K+1;k++) 
             for (i=max(1,wl-k*l-(k-1)*tr+1);i<=wl;i++)
		 mbest[i*(K+2)+k] = 1.0;
          for (k=1; k<=K+1; k++)
             for (i=wl-k*l-(k-1)*tr; i>0; i--)
		 { cost = (pcost[i*L+l] < l+1) ?
		            pcost[i*L+l]/(l-pcost[i*L+l]+1) : 1.0;
		   if (cost > 1.0) cost = 1.0;
                   if (k==1) this = cost;
		   else this = 1-(1-cost)*(1-mbest[(i+l+tr)*(K+2)+(k-1)]);
                   ibest[i*(K+2)+k] = i;
                   if (mbest[(i+1)*(K+2)+k] < this)
                      { this = mbest[(i+1)*(K+2)+k];
                        ibest[i*(K+2)+k] = ibest[(i+1)*(K+2)+k];
                      }
                   mbest[i*(K+2)+k] = this;
                 }
          if (mbest[1*(K+2)+(K+1)] < best)  /* put the results in output */
             { *wlen = l;
               i = 1;
               for (k=K+1;k>=1;k--)
                  { COPY(dinitial[K+1-k],initials[ibest[i*(K+2)+k]],m);
                    i = ibest[i*(K+2)+k]+l+tr;
                  }
               best = mbest[1*(K+2)+(K+1)];
             }
        }
     free (pcost); free (mbest); free (ibest);
     free (initials[0]); free (initials);

     if (best >= BK) { *wlen = 0; return 1.0; }

	/* there is a promising k+1 splitting. select the appropriate states */

	/* compute dactive and dfinal */

     tmp = createMask(m);
     dfinal[0] = createMasks (K+1,m);
     dactive[0] = createMasks (K+1,m);
     for (k=0;k<=K;k++)
	{ dfinal[k] = dfinal[0] + k*maskSize(m);
	  dactive[k] = dactive[0] + k*maskSize(m);
	  COPY(dactive[k],dinitial[k],m);
	  for (i=0;i<*wlen;i++)
             { ZERO(tmp,m);
               for (j=0;j<m;j++)
                  if (ISSET(dactive[k],j)) OR (tmp,trans[j][s],m);
               OR(dactive[k],tmp,m);
	     }
	  COPY(dfinal[k],tmp,m);
        }
     free (tmp);

                /* make ralltrans[i] = reverse of union trans[i][c] for all c,
                   O(m^2*s/w + m^2) time */
    
     ralltrans = malloc (m*sizeof(Mask));
     ralltrans[0] = createMasks(m,m);
     for (i=0;i<m;i++)
        ralltrans[i] = ZERO(ralltrans[0]+i*maskSize(m),m);
     for (i=0;i<m;i++)
        for (j=0;j<m;j++)
           if (ISSET(trans[i][s],j)) SET(ralltrans[j],i);

	/* compute b* and f* masks, O(k*m^3/w) */

     binitial[0] = createMasks (K+1,m);
     bfinal[0] = createMasks (K+1,m);
     bactive[0] = createMasks (K+1,m);
     finitial[0] = createMasks (K+1,m);
     ffinal[0] = createMasks (K+1,m);
     factive[0] = createMasks (K+1,m);
     for (k=0;k<=K;k++)
	{ binitial[k] = binitial[0] + k*maskSize(m);
	  bfinal[k] = bfinal[0] + k*maskSize(m);
	  bactive[k] = bactive[0] + k*maskSize(m);
	  finitial[k] = finitial[0] + k*maskSize(m);
	  ffinal[k] = ffinal[0] + k*maskSize(m);
	  factive[k] = factive[0] + k*maskSize(m);
	  regularComplete (*wlen,m,s+1,initial,final,trans,ralltrans,
                           dinitial[k], dactive[k], dfinal[k],
                           binitial[k], bactive[k], bfinal[k],
                           finitial[k], factive[k], ffinal[k]);
	}

     return best;
   }

void eregularFreeScan (eregularScanData *scan)

   { regularFreeScan (scan->edata); 
     if (scan->type == KPLUS1) free (scan->final);
     free (scan->V3);
     free (scan);
   }

                /* loads masks for fast bwd/fwd */

eregularScanData *eregularLoadFast (int m, Mask **trans, int s, int wlen, 
				   Mask *B, Mask *active, Mask *initial, 
				   Mask *final, int *chMap, int k, int type)

        /* findBest ensures that this always fits in a single mask */

   { eregularScanData *scan = malloc (sizeof(eregularScanData));
     Mask uactive,uinitial,ufinal;
     int i,j,*map,width;
     Mask **redtrans,redinitial,redfinal;
     Mask **revtrans,revinitial,revfinal;

     scan->k = k;
     scan->s = s;
     scan->type = type;
     if (type != KPLUS1)
        { scan->edata = regularLoadFast(m,trans,s,wlen,B,*active,*initial,
					*final,chMap,true);
	}
     else
	{ uactive = ZERO(createMask(m),m);
	  uinitial = ZERO(createMask(m),m);
	  ufinal = ZERO(createMask(m),m);
	  for (i=0;i<=k;i++)
	     { OR(uactive,active[i],m);
	       OR(uinitial,initial[i],m);
	       OR(ufinal,final[i],m);
	     }
          scan->edata = regularLoadFast(m,trans,s,wlen,B,
					uactive,uinitial,ufinal,chMap,true);
          scan->edata->wlen = wlen;
          scan->edata->chMap = chMap;
		/* remap the states */
          regularRemapStates (trans,m,s,uinitial,ufinal,uactive,B,&redtrans,
		       &scan->edata->dm,&width,&scan->edata->slices,&redinitial,
		       &redfinal,chMap,&map,true);
	  scan->dm = scan->edata->dm;
	  free (uactive); free(uinitial); free(ufinal);
		/* remap the initial states (final of reverse) */
	  scan->final = malloc ((k+1)*sizeof(mask));
	  for (i=0;i<=k;i++)
	     { scan->final[i] = ZEROS;
	       for (j=0;j<m;j++)
                  if (ISSET(initial[i],j)) scan->final[i] |= 1<<map[j];
	     }
	  free (map);
                /* reverse arrows */
          regularReverseArrows (redtrans,scan->dm,s,redinitial,redfinal,
                         &revtrans,&revinitial,&revfinal);
          free (redtrans[0][0]); free (redtrans[0]); free (redtrans);
          free (redinitial); free (redfinal);
                /* make all states initial */
          for (j=0;j<scan->dm;j++) SET(revinitial,j);
                /* make deterministic */
          regularMakeDet1 (width,revtrans,scan->dm,s,&scan->edata->dtrans);
          scan->edata->dinitial = revinitial[0]; 
	  scan->edata->dfinal = revfinal[0];
          free (revtrans[0][0]); free (revtrans[0]); free (revtrans);
          free (revinitial); free (revfinal);
	}
     scan->V3 = malloc (2*(k+1)*sizeof(mask));
     return scan;
   }

eregularData *eregularPreproc (char *pat, Tree *tree, Tree **pos, int k)

   { eregularData *P = malloc (sizeof(eregularData));
     int i,c,wlen,wlen1,slices;
     Mask **trans,ninitial,nfinal;
     Mask *active,active1,*initial,initial1,*final,final1;
     Mask *activef,activef1,*initialf,initialf1,*finalf,finalf1;
     Mask *activeb,activeb1,*initialb,initialb1,*finalb,finalb1;
     Mask **redtrans,redinitial,redfinal;
     Mask **revtrans,revinitial,revfinal;
     Mask B[256],S[256],A;
     double best1,best2;
     int j,K,*beg,*end,*map;

	/* allocate and load the masks */

     P->m = 0;
     regularLength(tree,&P->m);
     ninitial = createMask(P->m);
     nfinal = createMask(P->m);
     B[0] = createMasks (256,P->m);
     S[0] = createMasks (256,P->m);
     for (c=0; c<256; c++)
         { B[c] = ZERO(B[0]+c*maskSize(P->m),P->m);
           S[c] = ZERO(S[0]+c*maskSize(P->m),P->m);
         }
     A = ZERO(createMask(P->m),P->m);
     
     regularLoadMasks (pat,strlen(pat),P->m,tree,pos,B,S,A,
                       &trans,ninitial,nfinal,P->chMap,&P->s);

	/* determine a subset of the states for the search */
     active = malloc ((k+1)*sizeof(Mask*));
     activeb = malloc ((k+1)*sizeof(Mask*));
     activef = malloc ((k+1)*sizeof(Mask*));
     initial = malloc ((k+1)*sizeof(Mask*));
     initialb = malloc ((k+1)*sizeof(Mask*));
     initialf = malloc ((k+1)*sizeof(Mask*));
     final = malloc ((k+1)*sizeof(Mask*));
     finalb = malloc ((k+1)*sizeof(Mask*));
     finalf = malloc ((k+1)*sizeof(Mask*));
     best1 = (k+1) * regularFindBest (tree,trans,P->m,P->s,P->chMap,B,ninitial,
				      nfinal,&wlen1,&active1,&initial1,&final1,
		      		      &activef1,&initialf1,&finalf1,
				      &activeb1,&initialb1,&finalb1,k);
     best2 = findBestMulti (tree,trans,P->m,P->s,P->chMap,B,ninitial,
                            nfinal,&wlen,active,initial,final,
                            activef,initialf,finalf,activeb,initialb,finalb,k);
     free (ninitial); free (nfinal);
     if ((best1 <= best2) || !wlen)  /* we prefer suffix */
	{ K = 0; wlen = wlen1; P->type = wlen ? BWD : FWD; 
	  active[0] = active1;
	  initial[0] = initial1;
	  final[0] = final1;
	  activef[0] = activef1;
	  initialf[0] = initialf1;
	  finalf[0] = finalf1;
	  activeb[0] = activeb1;
	  initialb[0] = initialb1;
	  finalb[0] = finalb1;
	}
     else
	{ P->type = KPLUS1; K = k;
          free (active1); free (initial1); free (final1);
          free (activef1); free (initialf1); free (finalf1);
          free (activeb1); free (initialb1); free (finalb1);
	}

        /* fourth, create the verification data, common proc */

     P->k = k;
     P->dfmV = malloc ((K+1)*sizeof(int));
     P->dbmV = malloc ((K+1)*sizeof(int));
     P->fwidthV = malloc ((K+1)*sizeof(int));
     P->bwidthV = malloc ((K+1)*sizeof(int));
     P->dftransV = malloc ((K+1)*sizeof(Mask***));
     P->dbtransV = malloc ((K+1)*sizeof(Mask***));
     P->finitialV = malloc ((K+1)*sizeof(Mask));
     P->ffinalV = malloc ((K+1)*sizeof(Mask));
     P->binitialV = malloc ((K+1)*sizeof(Mask));
     P->bfinalV = malloc ((K+1)*sizeof(Mask));

     for (i=0;i<=K;i++)
	{      /* create forward verification automaton */
          regularRemapStates (trans,P->m,P->s,initialf[i],finalf[i],activef[i],
		       B,&redtrans,&P->dfmV[i],&P->fwidthV[i],&slices,
		       &P->finitialV[i], &P->ffinalV[i],P->chMap,&map,true);
	  free (map);
          regularMakeDet (P->fwidthV[i],redtrans,P->dfmV[i],P->s,
			  &P->dftransV[i]);
          free (redtrans[0][0]); free (redtrans[0]); free (redtrans); 
	       /* create backward verification automaton */
          regularRemapStates(trans,P->m,P->s,initialb[i],finalb[i],activeb[i],B,
		       &redtrans,&P->dbmV[i],&P->bwidthV[i],&slices,&redinitial,
		       &redfinal,P->chMap,&map,true);
	  free (map);
          regularReverseArrows (redtrans,P->dbmV[i],P->s,redinitial,redfinal,
			 &revtrans,&P->binitialV[i],&P->bfinalV[i]);
          free (redtrans[0][0]); free (redtrans[0]); free (redtrans); 
          free (redinitial); free (redfinal);
          regularMakeDet (P->bwidthV[i],revtrans,P->dbmV[i],P->s,
			  &P->dbtransV[i]);
          free (revtrans[0][0]); free (revtrans[0]); free (revtrans); 
	}
     free (initialf[0]); free (finalf[0]); free (activef[0]);
     free (initialf); free (finalf); free (activef);
     free (initialb[0]); free (finalb[0]); free (activeb[0]);
     free (initialb); free (finalb); free (activeb);

	/* create searching subexpression according to the type of search */
     switch (detClass(tree,active[0],P->m))
	{ case SIMPLE:
	     beg = malloc ((k+1)*sizeof(int));
	     for (j=0;j<=k;j++)
	        { for (i=0;i<P->m;i++) if (ISSET(active[j],i)) break;
		  beg[j] = i;
		}
	     P->scanData = esimpleLoadFast(wlen,k,P->type,B,beg);
	     free (beg);
	     P->scanText = (escanProc)esimpleScan;
	     P->scanFree = (freeScanProc)esimpleFreeScan;
	     break;
	  case EXTENDED:
	     beg = malloc ((k+1)*sizeof(int));
	     for (j=0;j<=k;j++)
	        { for (i=0;i<P->m;i++) if (ISSET(active[j],i)) break;
		  beg[j] = i;
	          for (i++;i<P->m;i++) if (!ISSET(active[j],i)) break;
		  end[j] = i;
		}
	     P->scanData = eextendedLoadFast(wlen,k,P->type,B,S,A,beg,end);
	     free (beg);
	     P->scanText = (escanProc)eextendedScan;
	     P->scanFree = (freeScanProc)eextendedFreeScan;
	     break;
	  case REGULAR:
	     P->scanData = eregularLoadFast(P->m,trans,P->s,wlen,B,active,
					    initial,final,P->chMap,k,P->type);
	     P->scanText = (escanProc)eregularScan;
	     P->scanFree = (freeScanProc)eregularFreeScan;
	     break;
        }

     free (B[0]); free (S[0]); free (A);
     free (initial[0]); free (final[0]); free (active[0]);
     free (initial); free (final); free (active);
     free (trans[0][0]); free (trans[0]); free (trans);
     P->V1 = createMasks (4,P->m);
     P->V2 = malloc (2*(k+1)*sizeof(Mask));
     P->V2[0] = createMasks (2*(k+1),P->m);
     for (i=0;i<2*(k+1);i++)
	P->V2[i] = P->V2[0] + i*maskSize(P->m);
     return P;
   }

void eregularFree (eregularData *P)

	/* Frees P */

   { int K,i;
     P->scanFree (P->scanData);
     K = (P->type == KPLUS1) ? P->k : 0;
     for (i=0;i<=K;i++)
	 { free (P->dftransV[i][0][0][0]); free (P->dftransV[i][0][0]); 
           free (P->dftransV[i][0]); free (P->dftransV[i]);
           free (P->finitialV[i]); free (P->ffinalV[i]);
           free (P->dbtransV[i][0][0][0]); free (P->dbtransV[i][0][0]); 
           free (P->dbtransV[i][0]); free (P->dbtransV[i]);
           free (P->binitialV[i]); free (P->bfinalV[i]);
	 }
     free (P->dftransV); free (P->dbtransV);
     free (P->finitialV); free (P->binitialV);
     free (P->binitialV); free (P->bfinalV);
     free (P->V1); free (P->V2[0]); free (P->V2);
     free (P);
   }

static byte *dirCheck (byte *ptr, byte *top, int inc, int *chMap, int s,
		       Mask ***dtrans, Mask initial, Mask final, int m, 
		       int width, Mask *current, Mask *ocurrent, Mask T,
		       Mask tmp, Mask otmp, Mask ntmp, int *K)

		/* directional check, forward or backward.
		   receives maximal error level in *K and returns there
                   the best k found, being the return value the last
                   character considered in that case (NULL if nothing
                   can be found with *K errors)  */
		/* assumes k>0 and that ptr needs to be shifted before 
		   reading */

   { int och,ch,i,f,j,n;
     int slices = (m+width-1)/width;
     int r,k = *K;
     byte *ret = NULL;
     s--;

           /* first remove trivial case m=1 */

     if (m == 1)
        { *K = 0;
          while (*K <= k)
             { if ((inc == -1) && recCheckLeftContext(ptr,top)) return ptr;
               if ((inc == +1) && recCheckRightContext(ptr+1,top+1)) return ptr;
               if (ptr == top) return NULL; ptr += inc;
               if (OptIns) (*K)++; else return NULL;
             }
          return NULL;
        }

		/* fill initial masks */
     COPY(current[0],initial,m);
     COPY(ocurrent[0],initial,m);
     for (i=1;i<=k;i++)
	{ COPY(tmp,current[i-1],m);
	  if (OptDel)
	     { f = 0;
	       for (r=0;r<slices;r++)
	          { OR(tmp,dtrans[r][SLICE(current[i-1],f,width)][s],m);
	            f += width;
	            if ((f+width-1)/W != f/W) f += W - (f % W);
	          }
	       COPY(current[i],tmp,m);
               COPY(ocurrent[i],tmp,m);
			/* here it is possible that k >= m */
               if (!ISZERO(AND(tmp,final,m),m))
                  { if (((inc == -1) && recCheckLeftContext(ptr,top)) ||
                        ((inc == +1) && recCheckRightContext(ptr+1,top+1)))
                       { *K = i; k = i-1; ret = ptr; }
                  }
	      }
	}

           /* now process until automaton dies. each time it finds
              P with i<=k errors, annote i and pos and reduce k */
	
     n = 0;
     while (true)
	{ if (ptr == top) return ret;  /* end of buffer */
	  ptr += inc; n++;
	  och = ch; ch = chMap[*ptr]; /* new char */
	     /* now compute new state in ntmp */
	  ZERO (ntmp,m);
	  f = 0;
	  for (r=0;r<slices;r++)
	     { OR(ntmp,dtrans[r][SLICE(current[0],f,width)][ch],m);
	       f += width;
	       if ((f+width-1)/W != f/W) f += W - (f % W);
	     }
          if (!ISZERO(AND(COPY(tmp,ntmp,m),final,m),m)) 
				/* found with zero errors, return */
             { if (((inc == -1) && recCheckLeftContext(ptr,top)) ||
                   ((inc == +1) && recCheckRightContext(ptr+1,top+1)))
                  { *K = 0; return ptr; }
	     }
	  COPY(otmp,current[0],m);
	  COPY(current[0],ntmp,m);
	  for (i=1;i<=k;i++)
	     { ZERO(tmp,m);
	       if (OptDel) 
		  { f = 0;
	            for (r=0;r<slices;r++)
	               { OR(tmp,dtrans[r][SLICE(ntmp,f,width)][s],m);
	                 f += width;
	                 if ((f+width-1)/W != f/W) f += W - (f % W);
	               }
		  }
	       if (OptIns) OR(tmp,otmp,m);
	       if (OptSubs)
		  { f = 0;
	            for (r=0;r<slices;r++)
	               { OR(tmp,dtrans[r][SLICE(otmp,f,width)][s],m);
	                 f += width;
	                 if ((f+width-1)/W != f/W) f += W - (f % W);
	               }
		  }
	       f = 0;
	       for (r=0;r<slices;r++)
	          { OR(tmp,dtrans[r][SLICE(current[i],f,width)][ch],m);
	            f += width;
	            if ((f+width-1)/W != f/W) f += W - (f % W);
	          }
	       if (OptTransp && (n >= 2))
		  { ZERO(T,m);
	            f = 0;
	            for (r=0;r<slices;r++)
	               { OR(T,dtrans[r][SLICE(ocurrent[i-1],f,width)][ch],m);
	                 f += width;
	                 if ((f+width-1)/W != f/W) f += W - (f % W);
	               }
	            f = 0;
	            for (r=0;r<slices;r++)
	               { OR(tmp,dtrans[r][SLICE(T,f,width)][och],m);
	                 f += width;
	                 if ((f+width-1)/W != f/W) f += W - (f % W);
	               }
		  }
	       COPY(ntmp,tmp,m);
               if (!ISZERO(AND(tmp,final,m),m)) /* match found, be stricter */
                  { if (((inc == -1) && recCheckLeftContext(ptr,top)) ||
                        ((inc == +1) && recCheckRightContext(ptr+1,top+1)))
                       { ret = ptr;
                         do { i--; COPY(tmp,current[i],m); }
                         while ((i >= 0) && !ISZERO(AND(tmp,final,m),m));
                         *K = i+1;
                         if (i == -1) return ptr; /* found with 0 errors */
                         k = i;
                       }
	          }
	       COPY(ocurrent[i-1],otmp,m);
	       COPY(otmp,current[i],m);
	       COPY(current[i],ntmp,m);
	     }
	  if (ISZERO(ntmp,m)) return ret;  /* automaton died, return best */
	}
   }

static bool checkMatch1 (eregularData *P, int i, byte *pos, 
			 byte *beg, byte *end)

        /* checkMatch that does not care about a transposition where the
           pattern is divided */

   { byte *ptrb,*ptrf;          /* ptr to match of selected subpattern */
     int kb,kf;

	/* first part: backward check */

     kb = P->k;
     ptrb = dirCheck (pos,beg,-1,P->chMap,P->s,P->dbtransV[i],P->binitialV[i],
		      P->bfinalV[i],P->dbmV[i],P->bwidthV[i],P->V2,
		      P->V2+(P->k+1),P->V1,P->V1+maskSize(P->m),
		      P->V1+2*maskSize(P->m),P->V1+3*maskSize(P->m),&kb);
     if (ptrb == NULL) return false;

	/* second part: forward check */

     kf = P->k-kb;
     ptrf = dirCheck (pos-1,end-1,+1,P->chMap,P->s,P->dftransV[i],
		      P->finitialV[i],P->ffinalV[i],P->dfmV[i],P->fwidthV[i],
		      P->V2,P->V2+(P->k+1),P->V1,P->V1+maskSize(P->m),
		      P->V1+2*maskSize(P->m),P->V1+3*maskSize(P->m),&kf);
     if (ptrf == NULL) return false;

     return true;   /* match in [ptrb,ptrf] */
   }

static bool checkMatch (eregularData *P, int i, byte *pos,
                        byte **beg, byte **end)

        /* Checks that an initially promising match starting at pos
           corresponds or not to a complete match. The buffer limits
           are *beg and *end, and in case of successful match we return
           there the boundaries of the match found. */

   { byte *rbeg,*rend,*obeg,*oend;
     bool ret;

                  /* start by knowing my surrounding record */

     if (P->type == FWD)  /* pos-1 is last position read */
        { recGetRecord (pos-1,*beg,*end,&rbeg,&rend,&obeg,&oend);
          if ((rbeg > pos-1) || (rend <= pos-1)) return false; /* overlaps */
        }
     else
        { recGetRecord (pos,*beg,*end,&rbeg,&rend,&obeg,&oend);
          if ((rbeg > pos) || (rend <= pos)) return false; /* overlaps */
        }

     ret = checkMatch1(P,i,pos,rbeg,rend);
     if (ret) { *beg = obeg; *end = oend; return true; }

       /* if the pattern is really split and transpositions are permitted
          a delicate problem may appear if the solution needs to transpose
          the characters at the split point. we solve the problem in a
          rather brutal but effective way */

     if (OptTransp && (P->dbmV[i] > 0) && (P->dfmV[i] > 0) &&
                      (pos < rend) && (pos-1 >= rbeg))
        { byte c = *pos;
          *pos = pos[-1];
          pos[-1] = c;
          ret = checkMatch1(P,i,pos,rbeg,rend);
          pos[-1] = *pos;
          *pos = c;
          if (ret) { *beg = obeg; *end = oend; return true; }
        }

     return false;
   }

static bool fwdScan11 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     mask ***dtrans, mask initial, register mask final,
		     register int s, register int *chMap)
 
   { register byte *pos,*top;	/* current and final text pointers */
     register int och,ch,c;
     register mask current0,current1;
     register mask ocurrent0,otmp;
     register mask **dtrans0 = dtrans[0];

     s--;
		/* fill initial masks */

     current0 = ocurrent0 = initial;
     current1 = current0 | dtrans0[current0][s];

           /* now traverse the text */
	
     pos = *beg; top = *end;
     while (pos < top)
	{ if ((c = *pos++) == OptRecChar) continue;
	  ch = chMap[c]; /* take new character */
	     /* now compute new state in tmp */
	  current1 = current0 | dtrans0[current0][s] | dtrans0[current1][ch];
	  current0 = dtrans0[current0][ch];
	  current1 |= dtrans0[current0][s];
          och = ch;
          while (pos < top)
	     { if (current1 & final)
	          {  /* selected part of pattern has matched, check all */
	            if (checkMatch (P,0,pos,beg,end)) return true;
                  }
	       if ((c = *pos++) == OptRecChar) break;
	       ch = chMap[c]; /* take new character */
	          /* now compute new state in tmp */
	       otmp = current0;
	       current0 = dtrans0[current0][ch];
	       current1 = otmp|dtrans0[current0|otmp][s]|dtrans0[current1][ch]|
	                  dtrans0[dtrans0[ocurrent0][ch]][och];
	       ocurrent0 = otmp;
               och = ch;
	     }
	}
     return false;
   }

static bool fwdScan12 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     mask ***dtrans, mask initial, register mask final,
		     register int s, register int *chMap)
 
   { register byte *pos,*top;	/* current and final text pointers */
     register int och,ch,c;
     register mask current0,current1,current2;
     register mask ocurrent0,ocurrent1,otmp,tmp;
     register mask **dtrans0 = dtrans[0];

     s--;
		/* fill initial masks */

     current0 = ocurrent0 = initial;
     current1 = ocurrent1 = current0 | dtrans0[current0][s];
     current2 = current1 | dtrans0[current1][s];

           /* now traverse the text */
	
     pos = *beg; top = *end;
     while (pos < top)
	{ if ((c = *pos++) == OptRecChar) continue;
	  ch = chMap[c]; /* take new character */
	     /* now compute new state in tmp */
	  current2 = current1 | dtrans0[current1][s] | dtrans0[current2][ch];
	  current1 = current0 | dtrans0[current0][s] | dtrans0[current1][ch];
	  current0 = dtrans0[current0][ch];
	  current1 |= dtrans0[current0][s];
	  current2 |= dtrans0[current1][s];
          och = ch;
          while (pos < top)
	     { if (current2 & final)
	          {  /* selected part of pattern has matched, check all */
	            if (checkMatch (P,0,pos,beg,end)) return true;
                  }
	       if ((c = *pos++) == OptRecChar) break;
	       ch = chMap[c]; /* take new character */
	          /* now compute new state in tmp */
	       otmp = current0;
	       current0 = dtrans0[current0][ch];
	       tmp = otmp | dtrans0[current0|otmp][s] | dtrans0[current1][ch] |
	             dtrans0[dtrans0[ocurrent0][ch]][och];
	       ocurrent0 = otmp;
	       otmp = current1;
	       current1 = tmp;
	       current2 = otmp|dtrans0[current1|otmp][s]|dtrans0[current2][ch]|
	                  dtrans0[dtrans0[ocurrent1][ch]][och];
	       ocurrent1 = otmp;
               och = ch;
	     }
	}
     return false;
   }

static bool fwdScan21 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     mask ***dtrans, mask initial, register mask final,
		     register int width, register int s, register int *chMap)
 
   { register mask aux;
     register byte *pos,*top;	/* current and final text pointers */
     register int och,ch,c;
     register mask this = (width == W) ? ONES : (ONE << width)-1;
     register mask current0,current1;
     register mask ocurrent0,otmp;
     register mask **dtrans0 = dtrans[0];
     register mask **dtrans1 = dtrans[1];

     s--;
		/* fill initial masks */

     current0 = ocurrent0 = initial;
     current1 = current0 |
	        dtrans0[current0&this][s] | dtrans1[current0>>width][s];

           /* now traverse the text */
	
     pos = *beg; top = *end;
     while (pos < top)
	{ if ((c = *pos++) == OptRecChar) continue;
	  ch = chMap[c]; /* take new character */
	     /* now compute new state in tmp */
	  aux = current0;
	  current0 = dtrans0[current0&this][ch] | dtrans1[current0>>width][ch];
	  aux |= current0;
	  current1 = current0 | dtrans0[aux&this][s] | dtrans1[aux>>width][s] |
	             dtrans0[current1&this][ch] | dtrans1[current1>>width][ch];
          och = ch;
          while (pos < top)
	     { if (current1 & final)
	          {  /* selected part of pattern has matched, check all */
	            if (checkMatch (P,0,pos,beg,end)) return true;
                  }
	       if ((c = *pos++) == OptRecChar) break;
	       ch = chMap[c]; /* take new character */
	          /* now compute new state in tmp */
	       otmp = current0;
	       current0 = 
		      dtrans0[current0&this][ch] | dtrans1[current0>>width][ch];
	       aux = otmp | current0;
	       current1 = otmp | dtrans0[aux&this][s]|dtrans1[aux>>width][s] |
	                dtrans0[current1&this][ch]|dtrans1[current1>>width][ch];
	       aux = dtrans0[ocurrent0&this][ch]|dtrans1[ocurrent0>>width][ch];
	       current1 |= dtrans0[aux&this][och]|dtrans1[aux>>width][och];
	       ocurrent0 = otmp;
               och = ch;
	     }
	}
     return false;
   }

static bool fwdScan22 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     mask ***dtrans, mask initial, register mask final,
		     register int width, register int s, register int *chMap)
 
   { register mask aux;
     register byte *pos,*top;	/* current and final text pointers */
     register int och,ch,c;
     register mask this = (width == W) ? ONES : (ONE << width)-1;
     register mask current0,current1,current2;
     register mask ocurrent0,ocurrent1,otmp,tmp;
     register mask **dtrans0 = dtrans[0];
     register mask **dtrans1 = dtrans[1];

     s--;
		/* fill initial masks */

     current0 = ocurrent0 = initial;
     current1 = ocurrent1 = current0 |
	        dtrans0[current0&this][s] | dtrans1[current0>>width][s];
     current2 = current1 |
	        dtrans0[current1&this][s] | dtrans1[current1>>width][s];

           /* now traverse the text */
	
     pos = *beg; top = *end;
     while (pos < top)
	{ if ((c = *pos++) == OptRecChar) continue;
	  ch = chMap[c]; /* take new character */
	     /* now compute new state in tmp */
	  otmp = current0;
	  current0 = tmp = dtrans0[otmp&this][ch] | dtrans1[otmp>>width][ch];
	  aux = tmp|otmp;
	  tmp = otmp | dtrans0[aux&this][s] | dtrans1[aux>>width][s]
	        | dtrans0[current1&this][ch] | dtrans1[current1>>width][ch];
	  otmp = current1;
	  current1 = tmp;
	  aux = tmp|otmp;
	  current2 = otmp | dtrans0[aux&this][s] | dtrans1[aux>>width][s]
	            | dtrans0[current2&this][ch] | dtrans1[current2>>width][ch];
          och = ch;
          while (pos < top)
	     { if (current2 & final)
	          {  /* selected part of pattern has matched, check all */
	            if (checkMatch (P,0,pos,beg,end)) return true;
                  }
	       if ((c = *pos++) == OptRecChar) break;
	       ch = chMap[c]; /* take new character */
	          /* now compute new state in tmp */
	       otmp = current0;
	       current0 = 
		      dtrans0[current0&this][ch] | dtrans1[current0>>width][ch];
	       aux = current0|otmp;
	       tmp = otmp | dtrans0[aux&this][s] | dtrans1[aux>>width][s] |
	             dtrans0[current1&this][ch] | dtrans1[current1>>width][ch];
	       aux = dtrans0[ocurrent0&this][ch]|dtrans1[ocurrent0>>width][ch];
	       tmp |= dtrans0[aux&this][och] | dtrans1[aux>>width][och];
	       ocurrent0 = otmp;
	       otmp = current1;
	       current1 = tmp;
	       aux = current1|otmp;
	       current2 = otmp | dtrans0[aux&this][s] | dtrans1[aux>>width][s] |
	             dtrans0[current2&this][ch] | dtrans1[current2>>width][ch];
	       aux = dtrans0[ocurrent1&this][ch]|dtrans1[ocurrent1>>width][ch];
	       current2 |= dtrans0[aux&this][och] | dtrans1[aux>>width][och];
	       ocurrent1 = otmp;
               och = ch;
	     }
        }
     return false;
   }

static bool fwdScanrk (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     register mask ***dtrans, mask initial, register mask final,
		     register int slices, register int width, register int k, 
		     register int s, register int *chMap, 
		     register mask *current, register mask *ocurrent)
 
   { register int c,i,r;
     register mask tmp,otmp,aux,aux2,aux3,T;
     register byte *pos,*top;	/* current and final text pointers */
     register int och,ch;
     register mask this = (width == W) ? ONES : (ONE << width)-1;

     s--;
		/* fill initial masks */

     tmp = current[0] = ocurrent[0] = initial;
     for (i=1;i<=k;i++)
	{ aux = tmp;
	  for (r=0;r<slices;r++)
	     { tmp |= dtrans[r][aux&this][s];
	       aux >>= width;
	     }
	  current[i] = ocurrent[i] = tmp;
	}

           /* now traverse the text */
	
     pos = *beg; top = *end;
     while (pos < top)
	{ if ((c = *pos++) == OptRecChar) continue;
	  ch = chMap[c]; /* take new character */
	     /* now compute new state in tmp */
	  otmp = aux = current[0];
	  tmp = ZEROS;
	  for (r=0;r<slices;r++) { tmp|=dtrans[r][aux&this][ch]; aux>>=width; }
	  current[0] = tmp;
	  for (i=1;i<=k;i++)
	     { aux = tmp|otmp; aux2 = current[i];
	       tmp = otmp;
	       for (r=0;r<slices;r++)
	          { tmp |= dtrans[r][aux&this][s] | dtrans[r][aux2&this][ch];
	            aux >>= width;
	          }
	       otmp = current[i];
	       current[i] = tmp;
             }
          och = ch;
          while (pos < top)
	     { if (tmp & final)
	          {  /* selected part of pattern has matched, check all */
	            if (checkMatch (P,0,pos,beg,end)) return true;
                  }
	       if ((c = *pos++) == OptRecChar) break;
	       ch = chMap[c]; /* take new character */
	          /* now compute new state in tmp */
	       otmp = aux = current[0];
	       tmp = ZEROS;
	       for (r=0;r<slices;r++) 
		   { tmp|=dtrans[r][aux&this][ch]; aux>>=width; }
	       current[0] = tmp;
	       for (i=1;i<=k;i++)
	          { aux = tmp|otmp; aux2 = current[i]; aux3 = ocurrent[i-1];
	            tmp = otmp; T = ZEROS;
	            for (r=0;r<slices;r++)
	               { tmp |= dtrans[r][aux&this][s]|dtrans[r][aux2&this][ch];
	                 T |= dtrans[r][aux3&this][ch];
	                 aux >>= width; aux2 >>= width; aux3 >>= width;
	               }
	            for (r=0;r<slices;r++) 
			{ tmp|=dtrans[r][T&this][och]; T>>=width;}
	            ocurrent[i-1] = otmp;
	            otmp = current[i];
	            current[i] = tmp;
                  }
               och = ch;
	     }
	}
     return false;
   }

static bool kplus1Scan1 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		  mask ***dtrans, register mask initial, register mask final, 
	          mask *finals, register int wlen, register int *chMap, int k)
 
   { register mask ocurrent,current;
     register byte *pos,*top;	/* current and final text pointers */
     register mask **dtrans0 = dtrans[0];
     register mask *dtransin = dtrans0[initial];
     register int j;
     int i;

     pos = *beg; top = *end;
     pos--;
     top -= wlen;
     while (pos < top)
        { if (!(current = dtransin[chMap[pos[wlen]]]))  /* skip loop */
	     { pos += wlen; continue; }
          j = wlen-1;
	  while (true)
	     { ocurrent = current;
	       current = dtrans0[current][chMap[pos[j]]];
	       if (!current) break;
	       if (--j == 0) 
	          {  /* a factor of the pattern has matched, 
			if it is a prefix, check all */
	            if (current & final)
		       {  /* determine which one(s) matched */
			 for (i=0;i<=k;i++)
			     if ((ocurrent & finals[i]) && 
			         checkMatch (P,i,pos+1,beg,end))
		                return true;
		       }
	 		  /* else shift in 1 */
	            j = 1;
		    break;
	          }
	      }
	  pos += j;
        }
     return false;
   }

static bool kplus1Scan2 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		  mask ***dtrans, register mask initial, register mask final, 
	          mask *finals, register int width, register int wlen, 
		  register int *chMap, int k)
 
   { register mask ocurrent,current;
     register byte *pos,*top;	/* current and final text pointers */
     register int ch;
     register mask this = (width == W) ? ONES : (ONE << width)-1;
     register mask **dtrans0 = dtrans[0];
     register mask **dtrans1 = dtrans[1];
     register int j;
     int i;

     pos = *beg; top = *end;
     pos--;
     top -= wlen;
     while (pos < top)
        { current = initial;
	  j = wlen;
	  while (true)
	     { ch = chMap[pos[j]]; /* take new character */
		/* compute transition */
	       ocurrent = current;
	       current = dtrans0[current&this][ch]|dtrans1[current>>width][ch];
	       if (!current) break;
	       if (--j == 0) 
	          {  /* a factor of the pattern has matched, 
			if it is a prefix, check all */
	            if (current & final)
		       {  /* determine which one(s) matched */
			 for (i=0;i<=k;i++)
			     if ((ocurrent & finals[i]) && 
			         checkMatch (P,i,pos+1,beg,end))
		                return true;
		       }
	 		  /* else shift in 1 */
	            j = 1;
		    break;
	          }
	      }
	  pos += j;
        }
     return false;
   }

static bool kplus1Scan (byte **beg, byte **end, echeckProc checkMatch, void *P,
		  mask ***dtrans, register mask initial, register mask final, 
	          mask *finals, register int slices, register int width,
		  register int wlen, register int *chMap, int k)
 
   { register mask ocurrent,current,tmp;
     register byte *pos,*top;	/* current and final text pointers */
     register int ch,i,j;
     register mask this = (width == W) ? ONES : (ONE << width)-1;

     pos = *beg; top = *end;
     pos--;
     top -= wlen;
     while (pos < top)
        { current = initial;
	  j = wlen;
	  while (true)
	     { ch = chMap[pos[j]]; /* take new character */
		/* compute transition */
	       ocurrent = tmp = current; current = ZEROS;
	       for (i=0;i<slices;i++)
	           { current |= dtrans[i][tmp&this][ch];
		     tmp >>= width;
	           }
	       if (!current) break;
	       if (--j == 0) 
	          {  /* a factor of the pattern has matched, 
			if it is a prefix, check all */
	            if (current & final)
		       {  /* determine which one(s) matched */
			 for (i=0;i<=k;i++)
			     if ((ocurrent & finals[i]) && 
			         checkMatch (P,i,pos+1,beg,end))
		                return true;
		       }
	 		  /* else shift in 1 */
	            j = 1;
		    break;
	          }
	      }
	  pos += j;
        }
     return false;
   }

static bool bwdScan11 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     mask ***dtrans, register mask initial, 
		     register mask final, register int wlen,
		     register int *chMap, register int s)
 
   { register byte *pos,*top;	/* current and final text pointers */
     register int och,ch,j;
     register mask current0,current1;
     register mask ocurrent0,otmp;
     register mask **dtrans0 = dtrans[0];

     s--;

		/* scan the text */
     pos = *beg; top = *end;
     pos--;
     wlen -= 1;
     top -= wlen;
     while (pos < top)
        {       /* fill initial masks (process one char now) */
	  och = ch = chMap[pos[wlen]]; /* take new character */
	  ocurrent0 = current1 = initial;
	  current0 = dtrans0[initial][ch];
	  j = wlen-1;
	  while (true)
	     { ch = chMap[pos[j]]; /* take new character */
		   /* compute transition */
	       otmp = current0;
	       current0 = dtrans0[current0][ch];
	       current1 = otmp|dtrans0[current0|otmp][s]|dtrans0[current1][ch]|
	                  dtrans0[dtrans0[ocurrent0][ch]][och];
	       ocurrent0 = otmp;
	       if (!current1) break;
	       if (--j == 0) 
	          {  /* a factor of the pattern has matched, 
			if it is a prefix, check all */
	            if ((current1 & final) && checkMatch (P,0,pos+1,beg,end))
		       return true;
	 		  /* else shift in 1 */
	            j = 1;
		    break;
	          }
               och = ch;
	     }
	  pos += j;
        }
     return false;
   }

static bool bwdScan21 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     mask ***dtrans, register mask initial, 
		     register mask final, register int width, register int wlen,
		     register int *chMap, register int s)
 
   { register mask aux,otmp;
     register byte *pos,*top;	/* current and final text pointers */
     register int och,ch,j;
     register mask this = (width == W) ? ONES : (ONE << width)-1;
     register mask current0,current1;
     register mask ocurrent0;
     register mask **dtrans0 = dtrans[0];
     register mask **dtrans1 = dtrans[1];

     s--;

		/* scan the text */
     pos = *beg; top = *end;
     pos--;
     wlen -= 2;
     top -= wlen;
     while (pos < top)
        {       /* fill initial masks (process one char now) */
	  och = ch = chMap[pos[wlen]]; /* take new character */
	  ocurrent0 = current1 = initial;
	  current0 = dtrans0[initial&this][ch] | dtrans1[initial>>width][ch];
	  j = wlen-1;
	  while (true)
	     { ch = chMap[pos[j]]; /* take new character */
		   /* compute transition */
	       otmp = current0;
	       current0 = dtrans0[current0&this][ch] | 
			  dtrans1[current0>>width][ch];
	       aux = current0|otmp;
	       current1 = otmp | dtrans0[aux&this][s] | dtrans1[aux>>width][s] |
	             dtrans0[current1&this][ch] | dtrans1[current1>>width][ch];
	       aux = dtrans0[ocurrent0&this][ch]|dtrans1[ocurrent0>>width][ch];
	       current1 |= dtrans0[aux&this][och] | dtrans1[aux>>width][och];
	       ocurrent0 = otmp;
	       if (!current1) break;
	       if (--j == 0) 
	          {  /* a factor of the pattern has matched, 
			if it is a prefix, check all */
	            if ((current1 & final) && checkMatch (P,0,pos+1,beg,end))
		       return true;
	 		  /* else shift in 1 */
	            j = 1;
		    break;
	          }
               och = ch;
	     }
	  pos += j;
        }
     return false;
   }

static bool bwdScan12 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     mask ***dtrans, register mask initial, 
		     register mask final, register int wlen,
		     register int *chMap, register int s)
 
   { register byte *pos,*top;	/* current and final text pointers */
     register int och,ch,j;
     register mask current0,current1,current2;
     register mask ocurrent0,ocurrent1,otmp,tmp;
     register mask **dtrans0 = dtrans[0];

     s--;

		/* scan the text */
     pos = *beg; top = *end;
     pos--;
     wlen -= 1;
     top -= wlen;
     while (pos < top)
        {       /* fill initial masks (process one char now) */
	  och = ch = chMap[pos[wlen]]; /* take new character */
	  ocurrent0 = ocurrent1 = current1 = current2 = initial;
	  current0 = dtrans0[initial][ch];
	  j = wlen-1;
	  while (true)
	     { ch = chMap[pos[j]]; /* take new character */
		   /* compute transition */
	       otmp = current0;
	       current0 = dtrans0[current0][ch];
	       tmp = otmp | dtrans0[current0|otmp][s] | dtrans0[current1][ch] |
	             dtrans0[dtrans0[ocurrent0][ch]][och];
	       ocurrent0 = otmp;
	       otmp = current1;
	       current1 = tmp;
	       current2 = otmp|dtrans0[current1|otmp][s]|dtrans0[current2][ch] |
	                  dtrans0[dtrans0[ocurrent1][ch]][och];
	       ocurrent1 = otmp;
	       if (!current2) break;
	       if (--j == 0) 
	          {  /* a factor of the pattern has matched, 
			if it is a prefix, check all */
	            if ((current2 & final) && checkMatch (P,0,pos+1,beg,end))
		       return true;
	 		  /* else shift in 1 */
	            j = 1;
		    break;
	          }
               och = ch;
	     }
	  pos += j;
        }
     return false;
   }

static bool bwdScan22 (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     mask ***dtrans, register mask initial, 
		     register mask final, register int width, register int wlen,
		     register int *chMap, register int s)
 
   { register mask aux,otmp,tmp;
     register byte *pos,*top;	/* current and final text pointers */
     register int och,ch,j;
     register mask this = (width == W) ? ONES : (ONE << width)-1;
     register mask current0,current1,current2;
     register mask ocurrent0,ocurrent1;
     register mask **dtrans0 = dtrans[0];
     register mask **dtrans1 = dtrans[1];

     s--;

		/* scan the text */
     pos = *beg; top = *end;
     pos--;
     wlen -= 2;
     top -= wlen;
     while (pos < top)
        {       /* fill initial masks (process one char now) */
	  och = chMap[pos[wlen]]; /* take new character */
	  ocurrent0 = ocurrent1 = current1 = current2 = initial;
	  current0 = dtrans0[initial&this][och] | dtrans1[initial>>width][och];
	  j = wlen-1;
	  while (true)
	     { ch = chMap[pos[j]]; /* take new character */
		   /* compute transition */
	       otmp = current0;
	       current0 = dtrans0[current0&this][ch] | 
			  dtrans1[current0>>width][ch];
	       aux = current0|otmp;
	       tmp = otmp | dtrans0[aux&this][s] | dtrans1[aux>>width][s] |
	             dtrans0[current1&this][ch] | dtrans1[current1>>width][ch];
	       aux = dtrans0[ocurrent0&this][ch]|dtrans1[ocurrent0>>width][ch];
	       tmp |= dtrans0[aux&this][och] | dtrans1[aux>>width][och];
	       ocurrent0 = otmp;
	       otmp = current1;
	       current1 = tmp;
	       aux = current1|otmp;
	       current2 = otmp | dtrans0[aux&this][s] | dtrans1[aux>>width][s] |
	             dtrans0[current2&this][ch] | dtrans1[current2>>width][ch];
	       aux = dtrans0[ocurrent1&this][ch]|dtrans1[ocurrent1>>width][ch];
	       current2 |= dtrans0[aux&this][och] | dtrans1[aux>>width][och];
	       ocurrent1 = otmp;
	       if (!current2) break;
	       if (--j == 0) 
	          {  /* a factor of the pattern has matched, 
			if it is a prefix, check all */
	            if ((current2 & final) && checkMatch (P,0,pos+1,beg,end))
		       return true;
	 		  /* else shift in 1 */
	            j = 1;
		    break;
	          }
               och = ch;
	     }
	  pos += j;
        }
     return false;
   }

static bool bwdScanrk (byte **beg, byte **end, echeckProc checkMatch, void *P,
		     register mask ***dtrans, register mask initial, 
		     register mask final, register int slices, 
		     register int width, register int wlen, register int *chMap,
		     register int s, register int k, register mask *current, 
		     register mask *ocurrent)
 
   { register mask tmp,otmp,aux,aux2,aux3,T;
     register byte *pos,*top;	/* current and final text pointers */
     register int och,ch,i,j,r;
     register mask this = (width == W) ? ONES : (ONE << width)-1;

     s--;

		/* scan the text */
     pos = *beg; top = *end;
     pos--;
     wlen -= k;
     top -= wlen;
     while (pos < top)
        {       /* fill initial masks (process one char now) */
	  och = chMap[pos[wlen]]; /* take new character */
	  ocurrent[0] = aux = initial;
	  tmp = ZEROS;
	  for (r=0;r<slices;r++)
	      { tmp |= dtrans[r][aux&this][och];
	        aux >>= width;
	      }
	  current[0] = tmp;
          for (i=1;i<=k;i++) current[i] = ocurrent[i] = initial;
	  j = wlen-1;
	  while (true)
	     { ch = chMap[pos[j]]; /* take new character */
		   /* compute transition */
	       otmp = aux = current[0];
	       tmp = ZEROS;
	       for (r=0;r<slices;r++)
	          { tmp |= dtrans[r][aux&this][ch];
	            aux >>= width;
	          }
	       current[0] = tmp;
	       for (i=1;i<=k;i++)
	          { aux = tmp|otmp; aux2 = current[i]; aux3 = ocurrent[i-1];
	            tmp = otmp; T = ZEROS;
	            for (r=0;r<slices;r++)
	                { tmp |= dtrans[r][aux&this][s] |
			         dtrans[r][aux2&this][ch];
	                  T |= dtrans[r][aux3&this][ch];
		          aux >>= width; aux2 >>= width; aux3 >>= width;
	                }
	            for (r=0;r<slices;r++)
	               { tmp |= dtrans[r][T&this][och];
	                 T >>= width;
	               }
	            ocurrent[i-1] = otmp;
	            otmp = current[i];
	            current[i] = tmp;
                  }
	       if (!tmp) break;
	       if (--j == 0) 
	          {  /* a factor of the pattern has matched, 
			if it is a prefix, check all */
	            if ((tmp & final) && checkMatch (P,0,pos+1,beg,end))
		       return true;
	 		  /* else shift in 1 */
	            j = 1;
		    break;
	          }
               och = ch;
	     }
	  pos += j;
        }
     return false;
   }

bool eregularScan (byte **beg, byte **end, echeckProc checkMatch, void *P,
		  eregularScanData *scan)
 
        /* Scans from *beg to *end-1 for P. Returns if it could find
           it. In case of returning true, *beg and *end are set to limit
           the first occurrence of P.
         */                                                                    

   { switch (scan->type)
	{ case KPLUS1:
	     switch (scan->edata->slices)
	        { case 1:
                  return kplus1Scan1 (beg,end,checkMatch,P,scan->edata->dtrans,
			            scan->edata->dinitial,scan->edata->dfinal,
			            scan->final,scan->edata->wlen,
			            scan->edata->chMap,scan->k);
	          case 2:
                  return kplus1Scan2 (beg,end,checkMatch,P,scan->edata->dtrans,
			            scan->edata->dinitial,scan->edata->dfinal,
			            scan->final,scan->edata->width,
				    scan->edata->wlen,scan->edata->chMap,
				    scan->k);
		  default:
                  return kplus1Scan (beg,end,checkMatch,P,scan->edata->dtrans,
			        scan->edata->dinitial,scan->edata->dfinal,
			        scan->final,scan->edata->slices,
				scan->edata->width,scan->edata->wlen,
				scan->edata->chMap,scan->k);
	        }
	  case FWD:
	     switch (scan->edata->slices)
	        { case 1:
		  switch (scan->k)
		     { case 1:
                       return fwdScan11 
			         (beg,end,checkMatch,P,scan->edata->dtrans,
			          scan->edata->dinitial,scan->edata->dfinal,
		                  scan->s,scan->edata->chMap);
		       case 2:
                       return fwdScan12 
			         (beg,end,checkMatch,P,scan->edata->dtrans,
			          scan->edata->dinitial,scan->edata->dfinal,
		                  scan->s,scan->edata->chMap);
		       default:
                       return fwdScanrk 
			      (beg,end,checkMatch,P,scan->edata->dtrans,
			       scan->edata->dinitial,scan->edata->dfinal,
		               scan->edata->slices,scan->edata->width,
			       scan->k,scan->s,scan->edata->chMap, 
		               scan->V3,scan->V3+(scan->k+1));
		     }
	          case 2:
		  switch (scan->k)
		     { case 1:
                       return fwdScan21 
				 (beg,end,checkMatch,P,scan->edata->dtrans,
			          scan->edata->dinitial,scan->edata->dfinal,
		                  scan->edata->width,scan->s,
				  scan->edata->chMap);
		       case 2:
                       return fwdScan22 
				 (beg,end,checkMatch,P,scan->edata->dtrans,
			          scan->edata->dinitial,scan->edata->dfinal,
		                  scan->edata->width,scan->s,
				  scan->edata->chMap);
		       default:
                       return fwdScanrk 
			      (beg,end,checkMatch,P,scan->edata->dtrans,
			       scan->edata->dinitial,scan->edata->dfinal,
		               scan->edata->slices,scan->edata->width,
			       scan->k,scan->s,scan->edata->chMap, 
		               scan->V3,scan->V3+(scan->k+1));
		     }
		  default:
                       return fwdScanrk 
			      (beg,end,checkMatch,P,scan->edata->dtrans,
			       scan->edata->dinitial,scan->edata->dfinal,
		               scan->edata->slices,scan->edata->width,
			       scan->k,scan->s,scan->edata->chMap, 
		               scan->V3,scan->V3+(scan->k+1));
	        }
	  case BWD: 
	     switch (scan->edata->slices)
	        { case 1:
		  switch (scan->k)
		     { case 1:
                       return bwdScan11 
			         (beg,end,checkMatch,P,scan->edata->dtrans,
			          scan->edata->dinitial,scan->edata->dfinal,
		                  scan->edata->wlen,scan->edata->chMap,scan->s);
		       case 2:
                       return bwdScan12 
			         (beg,end,checkMatch,P,scan->edata->dtrans,
			          scan->edata->dinitial,scan->edata->dfinal,
		                  scan->edata->wlen,scan->edata->chMap,scan->s);
		       default:
                       return bwdScanrk 
			      (beg,end,checkMatch,P,scan->edata->dtrans,
			       scan->edata->dinitial,scan->edata->dfinal,
		               scan->edata->slices,scan->edata->width,
			       scan->edata->wlen,scan->edata->chMap,scan->s,
		               scan->k,scan->V3,scan->V3+(scan->k+1));
		     }
	          case 2:
		  switch (scan->k)
		     { case 1:
                       return bwdScan21 
				 (beg,end,checkMatch,P,scan->edata->dtrans,
			          scan->edata->dinitial,scan->edata->dfinal,
		                  scan->edata->width,scan->edata->wlen,
				  scan->edata->chMap,scan->s);
		       case 2:
                       return bwdScan22 
				 (beg,end,checkMatch,P,scan->edata->dtrans,
			          scan->edata->dinitial,scan->edata->dfinal,
		                  scan->edata->width,scan->edata->wlen,
				  scan->edata->chMap,scan->s);
		       default:
                       return bwdScanrk 
			      (beg,end,checkMatch,P,scan->edata->dtrans,
			       scan->edata->dinitial,scan->edata->dfinal,
		               scan->edata->slices,scan->edata->width,
			       scan->edata->wlen,scan->edata->chMap,scan->s,
		               scan->k,scan->V3,scan->V3+(scan->k+1));
		     }
		  default:
                       return bwdScanrk 
			      (beg,end,checkMatch,P,scan->edata->dtrans,
			       scan->edata->dinitial,scan->edata->dfinal,
		               scan->edata->slices,scan->edata->width,
			       scan->edata->wlen,scan->edata->chMap,scan->s,
		               scan->k,scan->V3,scan->V3+(scan->k+1));
	        }
	}
   }

bool eregularSearch (byte **beg, byte **end, eregularData *P)

	/* Searches from *beg to *end-1 for P. Returns if it could find
	   it. In case of returning true, *beg and *end are set to limit
	   the first occurrence of P */

   { return P->scanText (beg,end,(echeckProc)checkMatch,P,P->scanData);
   }
 
