#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <dirent.h>
#include "lzgrep.h"
#include "search.h"
#define DIRSEP '/'


char *progname;
int numk;

/* List of patterns */
TypePatternList PatternList;

/* List of compressed files */
TypeFileList FileList;

static int redirZgrep = 0; /* Redirect to zgrep to search */

/* Global Options, -1 means unset */
int optContextBefore = 0; /* Number of context lines before a match */
int optContextAfter = 0;  /* Number of context lines after a match */
int optPrintFilename = -1; /* Prints name of matched file */
int optBinaryFiles = binaryFilesBinary; /* How to manipulate binary files */
int optByteOffset = 0; /* Print the byte offset */
int optCountMatches = 0; /* Only count the matches */
int optIgnoreCase = 0; /* Ignore case */
int optListFiles = 0; /* List files with match (=1), without match (=-1) */
int optStopAfter = 0; /* Stop after optStop After matchs (0 ==> no stop) */
int optLineNumber = 0;/* Prints line number of each matched line */
int optOnlyMatching = 0; /* Prints the only matching part of the line */
int optSilent = 0; /* Supress normal output */
int optSupressErrors = 0; /* Supress error messages */
char *optStdinLabel = NULL;/* Label for standard input */
int optDirAction = 0; /* Directories action. 0=skip 1=recurse */
int optDevAction = 0; /* Devices action 0=skip 1=read */

/* Global Indicators */
int indPrintMatches = 0;
int indLinePrefix = 0;

/* Parameters for current  search */
int fdSearch;  /* File Descriptor of ZFILE to search */
char *filenameSearch; /* Filename of file to search */




/* Long options with no short equivalence */
enum {
    ALGORITHM_OPTION = 128,
    BINARY_FILES_OPTION,
    HELP_OPTION,
    LABEL_OPTION
};

/* Equivalences between long an short options */
typedef struct equiv {
    char *longopt;
    unsigned char shortopt;
} TypeLongShortEquiv;

static TypeLongShortEquiv LongShortEquiv [] = {
    {"after-context",'A'},
    {"text",'a'},
    {"alg",ALGORITHM_OPTION},
    {"before-context",'B'},
    {"context",'C'},
    {"byte-offset",'b'},
    {"binary-files",BINARY_FILES_OPTION},
    {"count",'c'},
    {"extended-regexp",'E'},
    {"regexp",'e'},
    {"fixed-strings",'F'},
    {"file",'f'},
    {"basic-regexp",'G'},
    {"with-filename",'H'},
    {"no-filename",'h'},
    {"help",HELP_OPTION},
    {"ignore-case",'i'},
    {"directories",'d'},
    {"devices",'D'},
    {"recursive",'r'},
    {"files-without-match",'L'},
    {"files-with-matches",'l'},
    {"max-count",'m'},
    {"line-number",'n'},
    {"only-matching",'o'},
    {"label",LABEL_OPTION},
    {"quiet",'q'},
    {"silent",'q'},
    {"no-messages",'s'},
    {"version",'V'},
    {"invert-match",'v'},
    {"word-regexp",'w'},
    {"line-regexp",'x'},
};

/* Valid Options */
typedef struct voptions{
    int isValid;
    int useArgument;
} TypeValidOption;
static TypeValidOption ValidOption[256];


/* Fill Valid Options */
static void fill_valid_options() {
    int i;
    int isValid;
    int useArgument;

    for (i=0;i<256;i++) {
        switch(i) {
            case    'A':
            case ALGORITHM_OPTION:
            case 'B':
            case 'C':
            case BINARY_FILES_OPTION:
            case 'e':
            case 'f':
            case 'm':
            case 'D':
            case 'd':
		case 'k':
            case LABEL_OPTION:
                isValid=1;
                useArgument=1;
                break; 
            case 'a':
            case 'b':
            case 'c':
            case 'E':
            case 'F':
            case 'G':
            case 'H':
            case 'h':
            case HELP_OPTION:
            case 'I':
            case 'i':
            case 'L':
            case 'l':
            case 'n':
            case 'o':
            case 'q':
            case 'R':
            case 'r':
            case 's':
            case 'V':
            case 'v':
            case 'w':
            case 'x':
                   isValid=1;
                   useArgument=0;
                   break;
            default:
                   isValid=0;
                   useArgument=0;
                   break;
        }
        ValidOption[i].isValid = isValid;
        ValidOption[i].useArgument = useArgument;
        }    

}
/* Initial or default values */
static void init_values() {
    fill_valid_options();
}


/* Usage */
static void usage() {
    fprintf(stderr,"Usage: %s [OPTION]... PATTERN [ZFILE]...\n",progname);
    fprintf(stderr,"Try '%s --help' for more information.\n",progname);
    exit(2);
}

/* Help */
static void help() {
    printf ("Usage: %s [OPTION]... PATTERN [ZFILE] ...\n", progname);
    printf("Search for PATTERN in each compressed ZFILE or standard input\n");
    printf("Example: %s -i 'hello world' menu.Z menu2.Z\n",progname);
    printf("\nPattern selection and interpetation:\n");
    printf ("  -E, --extended-regexp     PATTERN is an extended regular expression (*)\n");
    printf ("  -F, --fixed-strings       PATTERN is a set of newline-separated strings\n");
    printf ("  -G, --basic-regexp        PATTERN is a basic regular expression (*)\n");
    printf ("  -P, --perl-regexp         PATTERN is a Perl regular expression (*)\n");
    printf ("  -e, --regexp=PATTERN      use PATTERN as a fixed pattern (**)\n");
    printf ("  -f, --file=FILE           obtain PATTERN from FILE\n");
    printf ("  -i, --ignore-case         ignore case distinctions\n");\
    printf ("  -w, --word-regexp         force PATTERN to match only whole words (*)\n");
    printf ("  -x, --line-regexp         force PATTERN to match only whole lines (*)\n");
    printf ("\nMiscellaneous:\n");
    printf ("  -s, --no-messages         suppress error messages\n");
    printf ("  -v, --invert-match        select non-matching lines (*)\n");
    printf ("  -V, --version             print version information and exit\n");
    printf ("      --help                display this help and exit\n");
    printf ("\nOutput control:\n");
    printf ("  -m, --max-count=NUM       stop after NUM matches\n");
    printf ("  -b, --byte-offset         print the byte offset with output lines\n");
    printf ("  -n, --line-number         print line number with output lines\n");
    printf ("  -H, --with-filename       print the filename for each match\n");
    printf ("  -h, --no-filename         suppress the prefixing filename on output\n");
    printf ("      --label=LABEL         print LABEL as filename for standard input\n");
    printf ("  -o, --only-matching       show only the part of a line matching PATTERN\n");
    printf ("  -q, --quiet, --silent     suppress all normal output\n");
    printf ("      --binary-files=TYPE   assume that binary files are TYPE\n");
    printf ("                            TYPE is 'binary', 'text', or 'without-match'\n");
    printf ("  -a, --text                equivalent to --binary-files=text\n");
    printf ("  -I                        equivalent to --binary-files=without-match\n");
    printf ("  -d, --directories=ACTION  how to handle directories\n");
    printf ("                            ACTION is 'recurse', or 'skip'\n");
    printf ("  -D, --devices=ACTION      how to handle devices, FIFOs and sockets\n");
    printf ("                            ACTION is 'read', or 'skip'\n");
    printf ("  -R, -r, --recursive       equivalent to --directories=recurse\n");
    printf ("  -L, --files-without-match only print FILE names containing no match\n");
    printf ("  -l, --files-with-matches  only print FILE names containing matches\n");
    printf ("  -c, --count               only print a count of matching lines per FILE\n");
    printf ("\nContext control:\n");
    printf ("  -B, --before-context=NUM  print NUM lines of leading context\n");
    printf ("  -A, --after-context=NUM   print NUM lines of trailing context\n");
    printf ("  -C, --context=NUM         print NUM lines of output context\n");
    printf ("\nSearch algorithm:\n");
    printf ("      --alg=D-algor            decompress-then-search using\n");
    printf ("                               algor = BM, BOM, KMP, WM, SBOM, AC\n");
    printf ("      --alg=BM-simple[-ext]    use BM-simple algorithm variant\n");
    printf ("                               ext = basic, lf, multi[-2|3|4]\n");
    printf ("      --alg=BM-multichar[-ext] use BM-multichar algorithm variant\n");
    printf ("                               ext = multi[-2]\n");
    printf ("      --alg=BM-blocks[-ext]    use BM-blocks algorithm variant\n");
    printf ("                               ext = basic, 2J, multi\n");
    printf ("\n(*) This option is implemented using zgrep (not lzgrep)\n");
    printf ("(**) Lzgrep always searches for fixed patterns\n");
    printf ("\nWith no ZFILE, or when ZFILE is -, read standard input.  If less than\n");
    printf ("two ZFILEs given, assume -h.  Exit status is 0 if match, 1 if no match,\n");
    printf ("and 2 if trouble.\n");

    exit(0);
}

/* Version */
static void version() {
   printf ("lzgrep 1.0\n");
   exit(0);
}

/* set context length argument */
static void set_context_length (char *str, int *out) {
    int value;
    int i, len;
    int is_valid, is_zero;

    len = strlen(str);
    is_valid = 1; 
    is_zero = 1;
    for (i=0; i < len; i++) {
	    if ((str[i] != '0') && is_zero) {
		    is_zero = 0;
	    } else if ((str[i] < '0') || (str[i] > '9')) {
		    is_valid = 0;
		    break;
	    }
    }
    if (is_valid) {
       value = atoi(str);
       if (!value && !is_zero) is_valid = 0;
    }

    if (!is_valid) {
	    fprintf(stderr,"%s: %s: invalid context length argument\n",
			    progname,str);
	    exit(2);
    } else {
	    *out = value;
    }
}


/* Add a string at the Pattern List, str is interpreted
 * a a list of patterns separated by newlines. This
 * function remember the position of the last added
 * pattern and add the new ones after it.
 */
static void add_to_pattern_list(char *str) {
        TypePatternNode *pnode = NULL;
	char *s = strdup(str);
	char *p;
	int len;
        
     while (1) {
	  while (*s == '\n') s++;
	  if (*s == '\0') break;
	  pnode = PatternList.last_pattern;
	  if (pnode) {
	      pnode->next_pattern = (TypePatternNode *)malloc(sizeof(TypePatternNode));	
	      pnode = pnode->next_pattern;
	  } else  {
	      pnode = (TypePatternNode *)malloc(sizeof(TypePatternNode));
	      PatternList.next_pattern = pnode;
	  }
	  PatternList.last_pattern = pnode;
	  pnode->next_pattern = NULL;
	  pnode->pattern = s;
	  len = strlen(s);
	  if (!PatternList.count) {
		  PatternList.lmin = len;
		  PatternList.lmax = len;
	  } else {
		  if (PatternList.lmin > len)  PatternList.lmin = len;
		  if (PatternList.lmax < len)  PatternList.lmax = len;
          }
	  PatternList.count++;
	  s = strchr(s,'\n');
	  if (s == NULL) break;
	  *s = '\0';
	  s++;
     }
}

/* Redirects to zgrep with the argument list argv,
 * argv[0] is replaced by "zgrep" and deletes
 * argv[i] if starts with "--alg=" (--alg is the only
 * one option of lzgrep not present in grep ).
 */
static void redirect_to_zgrep(int argc, char **argv) {
  char **zargv;
  int i;

  zargv = (char **) malloc(argc * sizeof(char *)+1);

  i=0;
  zargv[i] = strdup("zgrep");
  while (--argc) {
	  if (strncmp((++argv)[0],"--alg=",6))
		  zargv[++i] = strdup(argv[0]);
  }
  zargv[++i] = NULL;
  
  execvp("zgrep",zargv); 

  /* If execv returns is because an error has ocurred */
  fprintf(stderr,"%s: Can't redirect to zgrep\n",progname);
  perror(zargv[0]);
  exit(2);

}

/* Number of files to search */
/* Add a filename at the File List. This
 * function remember the position of the last added
 * filename and add the new ones after it.
 * If filename is "-" (standard input) set 
 * NULL to filename in file list
 */
static void add_to_file_list(char *filename) {
        TypeFileNode *fnode = NULL;
	char *fname = (strcmp(filename,"-") ? strdup(filename) : NULL);

	FileList.count++;
	fnode = FileList.last_file;
	if (fnode) {
	    fnode->next_file =(TypeFileNode *) malloc(sizeof(TypeFileNode));
	    fnode = fnode->next_file;
	} else {
	    fnode =(TypeFileNode *) malloc(sizeof(TypeFileNode));
	    FileList.next_file = fnode;
	}
	fnode->file = fname;
	fnode->next_file = NULL;
	FileList.last_file = fnode;

}

int process_file_argument(char *name) {
	struct stat stbuf; 
	DIR *d;
	struct dirent *dn;
	int rc, rc2;

	if (name != NULL) {
	   if (stat(name,&stbuf) < 0) {
		fprintf(stderr,"%s:",progname);
		perror(name);
		return(2);
	   }
	
	   if (S_ISDIR(stbuf.st_mode)) {
		if (optDirAction == 0) { /* skip */
		    if (optCountMatches && optPrintFilename)
			    printf("%s:0\n",name);
		    return(1);
		} else { /* read*/
			d = opendir(name);
			if (d==NULL) {
		           fprintf(stderr,"%s:",progname);
		           perror(name);
			   return(2);
			}
			rc = 1;
			while ((dn=readdir(d))>0) {
				if ((strcmp(dn->d_name,".")==0) || (strcmp(dn->d_name,"..") == 0))
					continue;
				rc2=process_file_argument(dn->d_name);
				if (rc2==0 && rc==1) rc=0;
				else if (rc2==2 && rc!=2) rc=2;
			}
			closedir(d);
			return(rc);
		}
	   } else if (S_ISCHR(stbuf.st_mode) ||S_ISBLK(stbuf.st_mode) || S_ISFIFO(stbuf.st_mode) || S_ISSOCK(stbuf.st_mode)) { /* is a device */
		if (optDevAction == 0) { /*skip*/
		    if (optCountMatches && optPrintFilename)
			    printf("%s:0\n",name);
		    return(1);
		}
           }
	}

	/* Search*/
        filenameSearch = (name ? name : optStdinLabel); 
	fdSearch = (name ? open(name,O_RDONLY) : 0);
	if (fdSearch < 0) {
	    fprintf(stderr,"%s: ",progname);
	    perror(name);
	    return(2);
	}
        rc = search();
	if (fdSearch > 0) close(fdSearch);
	return(rc);
}

/* Add patterns to pattern list from file filename. */
#define MAX_LINE 4096
static int add_patterns_from_file(char *filename) {
	FILE *fp;
	char line[MAX_LINE];
	int len;

	fp = fopen(filename,"r");
	if (fp == NULL) {
		fprintf(stderr,"%s: ",progname);
		perror(filename);
		return(-1);
	}
	while (fgets(line,MAX_LINE,fp) != NULL) {
             len = strlen(line);
	     if (line[len-1] == '\n') line[--len] = '\0';
	     if (line[len-1] == '\r') line[--len] = '\0';
	     if (len) add_to_pattern_list(line);
	}
	fclose(fp);
	return(0);
}

/* Get value for set of optStopAfter,
 * returns 0 if error ocurrs, else return the value
 * to use to set optStopAfter */
int get_stop_after(char *str) {
	int i,len;

	len = strlen(str);
	for (i=0; i<len; i++) {
		if (str[i] < '0' || str[i] > '9') return(0);
	}

	return(atoi(str));
}

int main(int argc, char **argv) {
    int i;
    unsigned char option;
    char *argument;
    TypeFileNode *fnode;
    TypePatternNode *pndode;
    enum { fileArg, patternArg, shortOptArg, longOptArg};
    int typeArg;
    char **zargv = argv;
    int zargc = argc;

    init_values();

    /* Program name */
    i=strlen(argv[0]);
    while((i--) && (argv[0][i]!=DIRSEP));
    progname = strdup(argv[0]+i+1);

    /* Options */
    if (argc < 2) usage();
    FileList.count = 0;
    FileList.next_file = NULL;
    PatternList.count = 0;
    PatternList.lmin = 0;
    PatternList.next_pattern = NULL;
    PatternList.last_pattern = NULL;
    while(--argc > 0) {
        /* Type of Argument */
        if (*((++argv)[0]) == '-')  {
	    switch((*argv)[1]) {
		    case '-':
                       typeArg = longOptArg;
		       break;
		    case '\0':
		       if (!PatternList.count) typeArg = patternArg;
		       else typeArg = fileArg;
		       break;
		    default:
                       typeArg = shortOptArg;
		       break;
            }
        } else if (!PatternList.count) {
            typeArg = patternArg;
        } else {
            typeArg = fileArg;
        }

        /* Convert long options to short */
        if (typeArg == longOptArg) {
               TypeLongShortEquiv *lsequiv;
               char *strArg = strdup(*argv+2);
               argument = strchr(strArg,'=');
               if (argument != NULL) {
                   argument[0] = '\0';
                   argument++;
               } 

               lsequiv = LongShortEquiv;  //mientras que no encuentre una coincidencia o 
               while ((lsequiv->longopt != NULL) && //no se termine la lista de opciones
                  strcmp(strArg,lsequiv->longopt))
                       lsequiv++;
               // si no se encontro alguna coincidencia mandar mensaje de error
               if (lsequiv->longopt == NULL) {
                   fprintf(stderr,"%s: unrecognized option '%s'\n",
                            progname,argv[0]);
                   usage();
               }
               
               option = lsequiv->shortopt;
               if (ValidOption[option].useArgument && argument == NULL) {
                   fprintf(stderr, "%s: option '%s' require an argument\n",
                     progname,argv[0]);
                   usage();
               } 
               if (!ValidOption[option].useArgument && argument != NULL) {
                   fprintf(stderr, 
                    "%s: option '--%s' doesn't allow an argument\n",
                    progname,strArg);
                   usage();
               }
        }

         /* Obtain argument for short option */
        if (typeArg == shortOptArg) {
            if ((*argv)[2] != '\0') {
                   fprintf(stderr,"%s: unrecognized option '%s'\n",
                            progname,argv[0]);
                   usage();
            }
            option = (*argv)[1];
            if (!ValidOption[option].isValid) {
                   fprintf(stderr,"%s: unrecognized option '%s'\n",
                            progname,argv[0]);
                   usage();
            }
	    if (ValidOption[option].useArgument) {
		   argument = (argv)[1];
                   if (argument==NULL) {
                      fprintf(stderr, 
                     "%s: option '%s' require an argument\n",
                     progname,argv[0]);
                      usage();
                   }
                   --argc;
		   ++argv;
            } else {
                  argument = NULL;
            }
            
        }

        /* Process Option */
        if (typeArg == fileArg) {
		/* Add a file to search list */
		add_to_file_list(argv[0]);
	}
        else if (typeArg == patternArg) {
		/* Add a pattern to search list */
		add_to_pattern_list(argv[0]);
	}
        else  switch(option) {
		/* Algorithm to search */
		case ALGORITHM_OPTION:
                  set_default_algorithm(argument);
			break;
		/* After context lines */
		case 'A':
			set_context_length(argument,&optContextAfter);
			break;
		/* Before context lines */
		case 'B':
			set_context_length(argument,&optContextBefore);
			break;
		/* Context lines */
		case 'C':
			set_context_length(argument,&optContextAfter);
			set_context_length(argument,&optContextBefore);
			break;
		/* How to Manipulate devices */
		case 'D':
			if (strcmp(argument,"skip")==0) 
                           optDevAction = 0;
			else if (strcmp(argument,"read")==0) 
                           optDevAction = 1;
			else {
			   fprintf(stderr,"%s: unknown devices method\n",
					   progname);
			   usage();
			}
			break;
		/* Option implemented redirecting to zgrep */
		case 'E':
			redirZgrep = 1;
			break;
		/* Fixed strings */ 
		case 'F':
			break;
		/* Options implemented redirecting to zgrep */
		case 'P':
		case 'G':
			redirZgrep = 1;
		        break;
		/* Prints filename for each matched file */
		case 'H':
			optPrintFilename = 1;
			break;
		/* Binary files without-match */
		case 'I':
			optBinaryFiles = binaryFilesWithoutMatch;
			break;
		/* Prints help */
		case HELP_OPTION: 
			help();
			break;
		/* Prints version */
		case 'V':
			version();
                        break;
		/* Binary Files text */
		case 'a':
			optBinaryFiles = binaryFilesText;
			break;
		/* Prints byte offset */
		case 'b':
			optByteOffset = 1;
			break;
		/* Count matches */
		case 'c':
			optCountMatches = 1;
			break;
		/* How to Manipulate directories */
		case 'd':

			if (strcmp(argument,"skip")==0) 
                           optDirAction = 0;
			else if (strcmp(argument,"recurse")==0) 
                           optDirAction = 1;
			else {
			   fprintf(stderr,"%s: unknown directories method\n",
					   progname);
			   usage();
			}
			break;
		/* Add pattern */
		case 'e':
		        add_to_pattern_list(argument);
			break;
		/* Add patterns from file */
		case 'f':
			if (add_patterns_from_file(argument) != 0)
				usage();
			break;
		/* No prints filename for matched files */
		case 'h':
			optPrintFilename = 0;
			break;
		/* Ignore case */
		case 'i':
			optIgnoreCase = 1;
			break;
		/* List files without match */
		case 'L':
			optListFiles = -1;
			break;
		/* List files with match */
		case 'l':
			optListFiles = 1;
			break;
		/* stop after option */
		case 'm':
			optStopAfter = get_stop_after(argument);
			if (!optStopAfter) usage();
			break;
		/* Prints line number of each matched line */
		case 'n':
			optLineNumber = 1;
			break;
		/* Prints only matching part of line */
		case 'o':
			optOnlyMatching = 1;
			break;
		/* Supress normal output */
		case 'q':
			optSilent = 1;
			break;
		/* Supress error messages */
		case 's':
			optSupressErrors = 1;
			break;
		/* Options implemented redirecting to zgrep */
		case 'w':
		case 'x':
			redirZgrep = 1;
		        break;
		/* Binary Files Option */
		case BINARY_FILES_OPTION:
			if (!strcmp(argument,"binary"))
				optBinaryFiles = binaryFilesBinary;
			else if (!strcmp(argument,"without-match"))
				optBinaryFiles = binaryFilesWithoutMatch;
			else if (!strcmp(argument,"text"))
				optBinaryFiles = binaryFilesText;
			else {
			    fprintf(stderr,"%s: unknown binary files type\n",
                                    progname);
			    usage();
		        }
			break;
		/* Label for standard input */
		case LABEL_OPTION:
		        if (optStdinLabel) free(optStdinLabel);
			optStdinLabel = strdup(argument);
			break;
		case 'k':
			 numk = atoi(argument);
			 break;
		/* Default */
		default:
			usage();

	}

    }

    if (!PatternList.count) usage();
    if (FileList.count == 0) add_to_file_list("-");
    if (!optStdinLabel) optStdinLabel = strdup("(standard input)");

    /* Set optPrintFilename to default */
    if (optPrintFilename == -1) {
        if (FileList.count > 1) optPrintFilename = 1;
	else optPrintFilename = 0;
    }

    /* Decide if redir to zgrep */
    if (redirZgrep) redirect_to_zgrep(zargc,zargv);

    /* Claculate global indicators */
    indPrintMatches = !optCountMatches && !optSilent && !optListFiles;
    indLinePrefix = indPrintMatches && (optByteOffset || optLineNumber || optPrintFilename);
    if (optListFiles) optStopAfter=1;

    /* Search files one by one */ 
    { int rc = 1;
      int rc2;
      fnode = FileList.next_file;
      while (fnode != NULL) {
	    rc2 = process_file_argument(fnode->file);
	    if (rc2==0 && rc==1) rc=0;
	    else if (rc2==2 && rc!=2) rc = 2;
	    fnode = fnode->next_file;
      };
      return(rc);
    }
}
