#include "format.h"
#include "zutil.h"
#include "search.h"

#include "search_bmsimple.h"
#include "search_bmsimple_multi.h"
#include "search_bmmultichar.h"
#include "search_bmmultichar_multi.h"
#include "search_bmblocks.h"
#include "search_bmblocks_multi.h"

#include "search_bm_onthefly.h" 
#include "search_bom_onthefly.h"
#include "search_kmp_onthefly.h"
#include "search_sbom_onthefly.h"
#include "search_ac_onthefly.h"



#include "lzgrep.h"
#include "blocks.h"
#include "util.h"
#include <errno.h>
#include <stdio.h>

static CompressFormat format;
static gzFile *gzSearchFile = NULL;

unsigned char SBUFF[DOBLE_BUF_SIZE];


struct algorithm {
	char *name;
	int is_multipattern;
	void (*LZW_preproc)(void);
	int  (*LZW_search)(void);
	void (*LZ77_preproc)(void);
	int  (*LZ77_search)(void);
};

enum { 
	D_BM_onthefly,
	D_KMP_onthefly,
	D_BOM_onthefly,
	D_WM_onthefly,
	D_AC_onthefly,
	D_SBOM_onthefly,
	BM_simple,
	BM_simple_opt,
	BM_simple_lf,
	BM_multichar,
	BM_blocks,
	BM_blocks_opt,
	BM_blocks_2j,
	BM_simple_multi,
	BM_simple_multi_2,
	BM_simple_multi_3,
	BM_simple_multi_4,
      BM_multichar_multi,
	BM_multichar_multi_2,
	BM_blocks_multi,
	null_search
};
struct algorithm AlgorithmList[] = {
      {"D-BM", 0, preproc_bm_onthefly,search_bm_onthefly,preproc_bm_onthefly,search_bm_onthefly},
      {"D-KMP", 0, preproc_kmp_onthefly,search_kmp_onthefly,preproc_kmp_onthefly, search_kmp_onthefly},
      {"D-BOM", 0, preproc_bom_onthefly,search_bom_onthefly,preproc_bom_onthefly,search_bom_onthefly},
      {"D-WM", 1,preproc_sbom_onthefly,search_sbom_onthefly, preproc_sbom_onthefly,search_sbom_onthefly}, // OJO WM not yet implemented, acts as SBOM
      {"D-AC", 1, preproc_ac_onthefly,search_ac_onthefly,preproc_ac_onthefly,search_ac_onthefly},
      {"D-SBOM", 1,preproc_sbom_onthefly,search_sbom_onthefly, preproc_sbom_onthefly,search_sbom_onthefly},
      {"BM-simple-basic", 0, preproc_bmsimple,search_bmsimple_1,NULL,NULL},
	{"BM-simple", 0, preproc_bmsimple,search_bmsimple_2,NULL,NULL},
	{"BM-simple-lf", 0, preproc_bmsimple,search_bmsimple_3,NULL,NULL},
      {"BM-multichar", 0, preproc_bmmultichar,search_bmmultichar,NULL,NULL},
	{"BM-blocks-basic", 0, preproc_bmblocks_1,search_bmblocks_1,NULL,NULL},
	{"BM-blocks", 0, preproc_bmblocks_2,search_bmblocks_2,NULL,NULL},
	{"BM-blocks-2J", 0, preproc_bmblocks_3,search_bmblocks_3,NULL,NULL},
	{"BM-simple-multi-2", 1, preproc_bmsimple_multi_1,search_bmsimple_multi_1,NULL,NULL},
	{"BM-simple-multi-3", 1, preproc_bmsimple_multi_2,search_bmsimple_multi_2,NULL,NULL},
	{"BM-simple-multi-4", 1, preproc_bmsimple_multi_3,search_bmsimple_multi_3,NULL,NULL},
	{"BM-simple-multi", 1, preproc_bmsimple_multi_4,search_bmsimple_multi_4,NULL,NULL},
      {"BM-multichar-multi-2", 1, preproc_bmmultichar_multi_1,search_bmmultichar_multi_1,NULL},
	{"BM-multichar-multi", 1, preproc_bmmultichar_multi_2,search_bmmultichar_multi_2,NULL},
	{"BM-blocks-multi", 1, preproc_bmblocks_multi,search_bmblocks_multi,NULL,NULL},
	{NULL, 0, NULL, NULL, NULL, NULL}
};
int indexDefaultAlg=null_search;
	
/* Set algorithm to search */
void set_default_algorithm(char *algname) {
	int i=0;

      while ((AlgorithmList[i].name != NULL)	&&
               strcmp(AlgorithmList[i].name,algname)) 
		   i++;
	if (AlgorithmList[i].name == NULL) {
		fprintf(stderr,"%s: invalid algorithm argument\n",progname);
		exit(2);
	}

	indexDefaultAlg = i;
}

/* Gets the index of the best algorithm */
static int get_best_algorithm(CompressFormat format) {
	int index;

	/* LZW format */
        if (format == LZW_format) {
	    /* Monopattern */
	    if (PatternList.count == 1)
                  index = BM_simple_opt;
	    /* Multipattern */
	    else if (PatternList.count <= 10)
		  index = BM_simple_multi_4;
	    else if (PatternList.count <= 100)
		  index = D_WM_onthefly;
	    else 
		  index = D_SBOM_onthefly;
        }
	/* LZ77 format */
	else {
	    /* Monopattern */
	    if (PatternList.count == 1)
                  index = D_BOM_onthefly;
	    /* Multipattern */
	    if (PatternList.count <= 100)
                  index = D_WM_onthefly;
	    else 
		  index = D_SBOM_onthefly;

	}

	return(index-1);
}

/* Selects algorithm to search
 * and do the search
 * Returns 0 ==> match, 1==> no match, 2==>troubles */

int search() {

      int n;
	int i;
	void (*fpreproc)(void);
	int  (*fsearch)(void);

	format = get_format();
       
	gzSearchFile = NULL;
	if (format == LZW_format) {
		 if (!codeopen()) return(2);
	}

	if (format == LZ77_format) {
		gzSearchFile = gzopenbuf(filenameSearch,fdSearch,inbuf,insize);
		if (!gzSearchFile) {
		        fprintf(stderr, "%s: %s: Corrupt file\n",
		            progname,filenameSearch);
			return(2);
		}
	}

	if (format == unsupported_format) {
		fprintf(stderr,
		  "%s: %s: invalid or unsupported compression format\n",
		  progname,filenameSearch);
		return(2);
	}
        
	i = indexDefaultAlg;
	if ((!AlgorithmList[i].is_multipattern) && (PatternList.count > 1)) {
		fpreproc = NULL;
		fsearch = NULL;
	} else if (format == LZW_format) {
		fpreproc = AlgorithmList[i].LZW_preproc;
		fsearch = AlgorithmList[i].LZW_search;
	} else {
		fpreproc = AlgorithmList[i].LZ77_preproc;
		fsearch = AlgorithmList[i].LZ77_search;
	}

	if (fpreproc == NULL || fsearch == NULL) {
		i = get_best_algorithm(format);
	        if (format == LZW_format) {
		   fpreproc = AlgorithmList[i].LZW_preproc;
		   fsearch = AlgorithmList[i].LZW_search;
	        } else {
		   fpreproc = AlgorithmList[i].LZ77_preproc;
		   fsearch = AlgorithmList[i].LZ77_search;
	        }
	}

       
	fpreproc();
	n = fsearch();
	if (optListFiles==1) {
		if (n) printf("%s\n",filenameSearch);
	} else if (optListFiles==-1) {
		if (!n) printf("%s\n",filenameSearch);
	} else if (optCountMatches) {
		if (optPrintFilename) printf("%s:",filenameSearch);
		printf("%d\n",n);
	}

	if (gzSearchFile) gzclose(gzSearchFile);
	if (n < 0) return(2);
	else if (n > 0) return(0);
	else return(1);

}

void OTF_cleanState(OTF_SearchState *OTF_State) {
	OTF_State->rpos = 0;
	OTF_State->tpos = 0;
	OTF_State->count = 0;
	OTF_State->ppos = 0;
	OTF_State->oldbuf = SBUFF;
	OTF_State->buf = SBUFF+BUF_SIZE;
	OTF_State->linenum = 1;
}

int OTF_readUncompressed(OTF_SearchState *OTF_State) {
	int rbytes;
	unsigned char *buf = OTF_State->oldbuf;
	
	OTF_State->oldbuf = OTF_State->buf;
	OTF_State->buf = buf;

	if (format == LZW_format) rbytes = getbuffer(buf,BUF_SIZE);
	else rbytes = gzread(gzSearchFile,buf,BUF_SIZE);

	if (rbytes > 0) OTF_State->rpos += rbytes;
	return(rbytes);
}


int OTF_reportMatch(OTF_SearchState *OTF_State) {
	int nlines;
	int maxlen, maxlen1, maxlen2;
	unsigned char  *p = &OTF_charAt(OTF_State->tpos);
	unsigned long ppos;
	int dif,i,len;
	int pos;
	int brk = 0;
	char sep;
	LineList  *list, *nel;
	int aux, head;


	OTF_State->count++;
	if (!indPrintMatches) goto OTF_afterMatch;
	if  ((OTF_State->count > 1)  && (optContextAfter || optContextBefore) && (OTF_State->ppos < OTF_State->tpos))
		printf("--\n");
	dif = p - OTF_State->buf;
	if ( (dif >= 0) && (dif < BUF_SIZE)) {
		maxlen1 = dif;
		maxlen2 = BUF_SIZE;
	} else {
		maxlen1 = p - OTF_State->oldbuf;
		maxlen2 = 0;
	}

	dif = OTF_State->tpos - OTF_State->ppos;
	if (dif < 0) dif = 0;
	if (dif < (maxlen1+maxlen2)) {
		if (maxlen1 > dif) {
			maxlen1 = dif;
			maxlen2 = 0;
		} else {
			maxlen2 = dif - maxlen1;
		}
	}

	/* Collect lines before match */
	nlines = optContextBefore +1;
	pos = OTF_State->tpos;
	list = NULL;
	for (i=0;i<2;i++) {
	   maxlen = (i?maxlen2:maxlen1);
	   len = 0;
           while (maxlen && nlines) {
		pos--;
		if (OTF_charAt(pos) == NEWLINE) {
			nel = (LineList *)malloc(sizeof(LineList));
			nel->buf = &OTF_charAt(pos+1);
		        nel->pos = pos+1;
			nel->len = len;
			nel->next = list;
			nel->pref = indLinePrefix;
			list = nel;
			len = 0;
			nlines--;
		}
		len++;
		--maxlen;
	   }
	   if (nlines && len) {
		nel = (LineList *) malloc(sizeof(LineList));
		nel->buf = &OTF_charAt(pos);
		nel->pos = pos;
		nel->len = len;
		nel->next = list;
		if (indLinePrefix) nel->pref = i;
		else nel->pref = 0;
		list = nel;
		aux = 1;
	   } else 
		aux = 0;
	}


	/* Print context lines before match */
	nel = list;
	nlines = optContextBefore;
	while (nel != NULL) {
		if (nel->pref) {
		      sep = (nlines?'-':':');
		      if (optPrintFilename) 
			      printf("%s%c",filenameSearch,sep);
		      if (optLineNumber)
			      printf("%d%c",OTF_State->linenum-nlines,sep);
		      if (optByteOffset)
			      printf("%d%c",nel->pos,sep);
		      nlines --;
		}
		printf("%.*s",nel->len,nel->buf);
		list = nel;
		nel = nel->next;
		free(list);
	}

	/* Print context lines after match */
OTF_afterMatch:
	if (indPrintMatches) nlines = optContextAfter+1;
	else nlines = 1;
	ppos = OTF_State->tpos;
	while (nlines > 0) {
		len = 0;
	        p = &OTF_charAt(ppos);
		if ((p>=OTF_State->oldbuf)  && (p < OTF_State->oldbuf+BUF_SIZE))
			maxlen = OTF_State->oldbuf+BUF_SIZE - p;
		else
		        maxlen = OTF_State->rpos - ppos;
		while((len < maxlen) && (OTF_charAt(ppos)!=NEWLINE)) {
			len++;
			ppos++;
		}
		if (OTF_charAt(ppos)==NEWLINE) {
			len++;
			ppos++;
		        nlines --;
			if (nlines == optContextAfter) {
			    OTF_State->tpos = ppos;
		            if (optLineNumber)
				OTF_State->linenum++;
			    head = indLinePrefix && aux;
			} else {
                            head = indLinePrefix;
			}
			if  (head) {
		            if (optPrintFilename) 
			            printf("%s-",filenameSearch);
		            if (optLineNumber)
			            printf("%d-",OTF_State->linenum+optContextAfter-nlines-1);
		            if (optByteOffset)
			            printf("%d-",ppos-len);
			}
		if (ppos > OTF_State->ppos)
		        if (indPrintMatches) printf("%.*s",len,p);
		len=0;
		} else if (((OTF_State->rpos - ppos) <= 0) && (OTF_readUncompressed(OTF_State)<=0)) {
			printf("<read nlines=%d>",nlines);
			if (nlines == (optContextAfter+1)) OTF_State->tpos = ppos;
			nlines = 0;
			brk = 1;
		}
	}
		if (ppos > OTF_State->ppos)
		if (indPrintMatches) printf("%.*s",len,p);
		OTF_State->ppos = ppos;

	if (optStopAfter && (OTF_State->count >= optStopAfter)) brk=1;	
	return(brk);

}

void OTF_shift2(OTF_SearchState *OTF_State,int shift) {
   int i,pos;
   for (i=0,pos=OTF_State->tpos;i<shift;pos++,i++) 
	   if (OTF_charAt(pos) == NEWLINE) OTF_State->linenum++;
   OTF_State->tpos = pos;
}
