

// Implements operations over a bitmap

#include "bitmap.h"

  // In theory, we should have superblocks of size s=log^2 n divided into
  // blocks of size b=(log n)/2. This takes 
  // O(n log n / log^2 n + n log log n / log n + log n sqrt n log log n) bits
  // In practice, we can have any s and b, and the needed amount of bits is
  // (n/s) log n + (n/b) log s + b 2^b log b bits
  // Optimizing it turns out that s should be exactly s = b log n
  // Optimizing b is more difficult but could be done numerically.
  // However, the exponential table does no more than popcounting, so why not
  // setting up a popcount algorithm tailored to the computer register size,
  // defining that size as b, and proceeding.

#define BX(x) ((x) - (((x)>>1)&0x77777777) - (((x)>>2)&0x33333333) - (((x)>>3)&0x11111111))

uint popcount (register int x)

   { x = BX(x);
     return (((x+(x>>4)) & 0x0F0F0F0F) % 255);
   }

	// creates a bitmap structure from a bitstring, which is shared

bitmap createBitmap (uint *string, uint n)

   { bitmap B;
     uint i,j,pop,bpop,pos;
     uint s,nb,ns,words;
     B = malloc (sizeof(struct sbitmap));
     B->data = string;
     B->n = n; words = (n+W-1)/W;
     B->nbits = bits(n-1);
     s = B->nbits*W;
     B->sbits = bits(s-1);
     s = 1 << B->sbits; // to take the most advantage of what we can represent
     ns = (n+s-1)/s; nb = (s+W-1)/W; // adjustments
     B->bdata = malloc (((words*B->sbits+W-1)/W)*sizeof(int));
     B->sdata = malloc (((ns*B->nbits+W-1)/W)*sizeof(int));
#ifdef INDEXREPORT
     printf ("     Bitmap over %i bits took %i bits\n",
		n,(words+((words*B->sbits+W-1)/W)+((ns*B->nbits+W-1)/W))*W);
#endif INDEXREPORT
     pop = 0; pos = 0;
     for (i=0;i<ns;i++)
	{ bpop = 0;
	  bitput (B->sdata,i*B->nbits,B->nbits,pop);
	  for (j=0;j<nb;j++)
	     { if (pos == words) break;
	       bitput (B->bdata,pos*B->sbits,B->sbits,bpop);
	       pos++;
	       bpop += popcount(*string++);
	     }
	  pop += bpop;
	}
     return B;
   }

	// rank(i): how many 1's are there before position i, not included

uint rank (bitmap B, uint i)

   { uint sblock,block,bpos;
     uint bitstr;
     bpos = i & ((1<<bitsW)-1);
     block = i >> bitsW;
     sblock = i >> B->sbits;
             // returns e[p..p+len-1], assuming len <= W
     return bitget(B->sdata,sblock*B->nbits,B->nbits) +
	    bitget(B->bdata,block*B->sbits,B->sbits) +
	    popcount (B->data[block] & ((1<<bpos)-1));
   }

	// destroys the bitmap, freeing the original bitstream

void destroyBitmap (bitmap B)

   { free (B->data);
     free (B->bdata);
     free (B->sdata);
     free (B);
   }
