require 'set'
require 'erb'
require 'json'
require 'net/http'
include ERB::Util

NATIONS = [
  "ALGERIA", "ARGENTINA", "BRAZIL", "CANADA", "EGYPT",
  "ETHIOPIA", "FRANCE", "GERMANY", "INDIA", "INDONESIA",
  "IRAN", "IRAQ", "JAPAN", "JORDAN", "KENYA", "MOROCCO",
  "MOZAMBIQUE", "PERU", "CHINA", "ROMANIA", "SAUDI ARABIA",
  "VIETNAM", "RUSSIA", "UNITED KINGDOM", "UNITED STATES"
]

[
  'CODE_HOME',
  'DB_DIR',
  'DATA_DIR',
  'VIRTUOSO_INI',
  'USER'
].each do |key|
  if ENV[key].nil?
    STDERR.puts "Error: ENV['#{key}'] is not defined."
    exit 1
  end
end

def read_range(range)
  begin
    r = eval(range)
    unless Range === r or Array === r
      puts "Range not valid: #{range}"
      exit 1
    end
  rescue SyntaxError => se
    puts "Range not valid: #{range}"
    exit 1
  end
  r
end

S_SIZE      = 10

class ParamGeneratorTPCH
  def initialize(scale_factor)
    @scale_factor = scale_factor
    @rg = Random.new 1
    @generated_nation    = Set.new
    @generated_countries = Set.new
    @generated_supplier  = Set.new
    @generated_color     = Set.new
  end

  def generate(key)
    params = {}

    if key == 'q1' or key == 'q4' # Nation
      loop do
        @nation = @rg.rand(25)
        break unless @generated_nation.include? @nation
      end
      @generated_nation << @nation
      params['?nation_param'] = "\"#{NATIONS[@nation]}\""
    end

    if key == 'q2' # Countries
      loop do
        @countries = Set.new
        7.times do
          loop do
            @country = @rg.rand(25)
            break unless @countries.include? @country
          end
          @countries << @country
        end
        break unless @generated_countries.include? @countries
      end
      @generated_countries << @countries
      params['?countries_param'] = @countries.map do |country|
        "\"#{NATIONS[country]}\""
      end.join(' , ')
    end

    if key == 'q3' # Supplier
      loop do
        @supplier = @rg.rand(@scale_factor*S_SIZE)
        break unless @generated_supplier.include? @supplier
      end
      @generated_supplier << @supplier
      params['?supplier_param'] = "supp:#{@supplier}"
    end

    if key == 'q4' # Color
      colors = [
        "almond", "antique", "aquamarine", "azure", "beige", "bisque", "black",
        "blanched", "blue", "blush", "brown", "burlywood", "burnished", "chartreuse",
        "chiffon", "chocolate", "coral", "cornflower", "cornsilk", "cream", "cyan",
        "dark",  "deep",  "dim", "dodger", "drab", "firebrick", "floral",  "forest",
        "frosted", "gainsboro", "ghost", "goldenrod", "green",  "grey",  "honeydew",
        "hot", "indian", "ivory", "khaki", "lace", "lavender", "lawn", "lemon",
        "light", "lime", "linen", "magenta", "maroon", "medium", "metallic",
        "mid night", "mint", "misty", "moccasin", "navajo", "navy", "olive", "orange",
        "orchid", "pale", "papaya", "peach", "peru", "pink", "plum", "powder", "puff",
        "purple", "red", "rose", "rosy", "royal", "saddle", "salmon", "sandy",
        "seashell",  "sienna", "sky", "slate", "smoke", "snow", "spring", "steel",
        "tan", "thistle", "tomato", "turquoise", "violet", "wheat", "white", "yellow"
      ]
      loop do
        @color = @rg.rand(92)
        break unless @generated_color.include? @color
      end
      @generated_color << @color
      #STDERR.puts @color.inspect
      params['?color_param'] = colors[@color]
    end

    params
  end
end

# Generate a query from a template.
def generate_query(template, params)
  query = template
  params.each { |k,v| query.gsub!(k,v) }
  query
end

# Run a query against an endpoint.
def run_query(endpoint, query, timeout)
  endpoint = URI(endpoint)
  http = Net::HTTP.new(endpoint.host, endpoint.port)
  http.open_timeout = 60
  http.read_timeout = timeout

  t1 = Time.now
  begin
    resp = http.post(endpoint, "query=#{url_encode(query)}", {
                       #'Content-Type'=>'application/sparql-query',
                       'Content-Type'=>'application/x-www-form-urlencoded',
                       'Accept'=>'application/sparql-results+json,application/json,*/*;q=0.9'})
    t2 = Time.now
    result = {
      'time'=> t2-t1,
      'query'=> query,
      'body'=> resp.body,
      'status'=> resp.code
    }
    begin
      result['doc'] = JSON.parse(resp.body)
    rescue
      result['error'] = 'truncated output'
    end
  rescue RuntimeError => e
    t2 = Time.now
    result = { 'time'=> t2-t1, 'status'=> 'timeout', 'error'=> e }
  end
  result
end

def run_query_until_success(endpoint, query, timeout)
  begin
    result = run_query(endpoint, query, timeout)
  rescue
    STDERR.write '.'
    sleep 2
    retry
  end
  result
end

class VirtuosoEndpoint
  attr_reader :endpoint

  def initialize
    @home = ENV['VIRTUOSO_HOME']
    @endpoint = ENV['VIRTUOSO_ENDPOINT']
  end

  def start(dbdir)
    dir = "#{ENV['VIRTUOSO_HOME']}/var/lib/virtuoso"
    db  = "#{dir}/db"
    cmd = "#{ENV['VIRTUOSO_HOME']}/bin/virtuoso-t"
    system "sudo -u #{ENV['USER']} rm #{db}" if File.exists? db
    system "sudo -u #{ENV['USER']} ln -s #{dbdir} #{db}"
    system "cd #{db}; sudo -u #{ENV['USER']} #{cmd}"
    STDERR.write 'Wait until server is accepting connections '
    sleep 2
    run_query_until_success(@endpoint, "ASK { ?s ?p ?o }", 10)
    STDERR.puts
  end

  def stop
    STDERR.write 'Closing server '
    system "pidof virtuoso-t | xargs kill"
    while !(`pidof virtuoso-t`.strip.empty?)
      sleep 2
      STDERR.write '.'
    end
    STDERR.puts
    system "rm #{ENV['VIRTUOSO_HOME']}/var/lib/virtuoso/db "
  end

  def load(file)
    tmp_file = "/tmp/load_data.sql"

    File.open(tmp_file, 'w') do |f|
      f.puts "ld_dir_all('#{File.dirname(file)}','#{File.basename(file)}','http://example.org');"
      f.puts "SELECT ll_file, ll_state FROM DB.DBA.LOAD_LIST;"
      f.puts "rdf_loader_run();"
      f.puts "SELECT ll_file, ll_state FROM DB.DBA.LOAD_LIST;"
      f.puts "checkpoint;"
      f.puts "commit work;"
      f.puts "checkpoint;"
      f.puts "DELETE FROM DB.DBA.LOAD_LIST;"
      f.puts "SPARQL SELECT (count(*) AS ?triples) " +
             "FROM <http://example.org>" +
             "WHERE { ?s ?p ?o };"
    end

    system "#{ENV['VIRTUOSO_HOME']}/bin/isql 1111 dba dba #{tmp_file}"
    system "rm #{tmp_file}"
    sleep 2
  end
end

class FusekiEndpoint
  attr_reader :endpoint

  def initialize
    @home     = ENV['FUSEKI_HOME']
    @endpoint = ENV['FUSEKI_ENDPOINT']
    @db       = "#{@home}/db"
  end

  def start(dbdir)
    system "rm #{@db}" if File.exists? @db
    system "ln -s #{dbdir} #{@db}"
    system ". #{ENV['CODE_CONFIG']}; cd #{@home}; ./fuseki start"
    STDERR.write 'Wait until server is accepting connections '
    sleep 2
    run_query_until_success(@endpoint, "ASK { ?s ?p ?o }", 10)
    STDERR.puts
  end

  def stop
    system "cd #{@home}; ./fuseki stop"
    sleep 5
    system "rm #{@db}"
  end
end
