#!/usr/bin/env ruby

require 'zlib'

if ARGV.size != 1
  STDERR.puts 'Usage: tbl_to_rdf <dir>'
  exit 1
end

def value(val, type)
  case type
  when :text
    val == '@' ? "[a <Blank>]" : "\"#{val}\""
  when :int
    val == '@' ? "[a <Blank>]" : "\"#{val}\"^^xsd:integer"
  when :decimal
    val == '@' ? "[a <Blank>]" : "\"#{val}\"^^xsd:decimal"
  when :date
    val == '@' ? "[a <Blank>]" : "\"#{val}\"^^xsd:date"
  when :region
    val == '@' ? "[a <Blank>]" : "<Region/#{val}>"
  when :nation
    val == '@' ? "[a <Blank>]" : "<Nation/#{val}>"
  when :part
    val == '@' ? "[a <Blank>]" : "<Part/#{val}>"
  when :supplier
    val == '@' ? "[a <Blank>]" : "<Supplier/#{val}>"
  when :customer
    val == '@' ? "[a <Blank>]" : "<Customer/#{val}>"
  when :order
    val == '@' ? "[a <Blank>]" : "<Order/#{val}>"
  else
    raise "Unknown type #{type}"
  end
end

def parse_row(row, model, file)
  case model
  when 'part'
    file.puts "<Part/#{row[0]}> a <Part> ;"
    file.puts "    <p_name> #{value(row[1], :text)} ;"
    file.puts "    <p_mfgr> #{value(row[2], :text)} ;"
    file.puts "    <p_brand> #{value(row[3], :text)} ;"
    file.puts "    <p_type> #{value(row[4], :text)} ;"
    file.puts "    <p_size> #{value(row[5], :int)} ;"
    file.puts "    <p_container> #{value(row[6], :text)} ;"
    file.puts "    <p_retailprice> #{value(row[7], :decimal)} ;"
    file.puts "    <p_comment> #{value(row[8], :text)} ."
  when 'region'
    file.puts "<Region/#{row[0]}> a <Region> ;"
    file.puts "    <r_name> #{value(row[1], :text)} ;"
    file.puts "    <r_comment> #{value(row[2], :text)} ."
  when 'nation'
    file.puts "<Nation/#{row[0]}> a <Nation> ;"
    file.puts "    <n_name> #{value(row[1], :text)} ;"
    file.puts "    <n_region> #{value(row[2], :region)} ;"
    file.puts "    <n_comment> #{value(row[3], :text)} ."
  when 'supplier'
    file.puts "<Supplier/#{row[0]}> a <Supplier> ;"
    file.puts "    <s_name> #{value(row[1], :text)} ;"
    file.puts "    <s_address> #{value(row[2], :text)} ;"
    file.puts "    <s_nation> #{value(row[3], :nation)} ;"
    file.puts "    <s_phone> #{value(row[4], :text)} ;"
    file.puts "    <s_acctbal> #{value(row[5], :decimal)} ;"
    file.puts "    <s_comment> #{value(row[6], :text)} ."
  when 'partsupp'
    file.puts "<PartSupp/#{row[0]}/#{row[1]}> a <PartSupp> ;"
    file.puts "    <ps_part> #{value(row[0], :part)} ;"
    file.puts "    <ps_supp> #{value(row[1], :supplier)} ;"
    file.puts "    <ps_availqty> #{value(row[2], :int)} ;"
    file.puts "    <ps_supplycost> #{value(row[3], :decimal)} ;"
    file.puts "    <ps_comment> #{value(row[4], :text)} ."
  when 'customer'
    file.puts "<Customer/#{row[0]}> a <Customer> ;"
    file.puts "    <c_name> #{value(row[1], :text)} ;"
    file.puts "    <c_address> #{value(row[2], :text)} ;"
    file.puts "    <c_nation> #{value(row[3], :nation)} ;"
    file.puts "    <c_phone> #{value(row[4], :text)} ;"
    file.puts "    <c_acctbal> #{value(row[5], :decimal)} ;"
    file.puts "    <c_mktsegment> #{value(row[6], :text)} ;"
    file.puts "    <c_comment> #{value(row[7], :text)} ."
  when 'orders'
    file.puts "<Order/#{row[0]}> a <Order> ;"
    file.puts "    <o_cust> #{value(row[1], :customer)} ;"
    file.puts "    <o_orderstatus> #{value(row[2], :text)} ;"
    file.puts "    <o_totalprice> #{value(row[3], :decimal)} ;"
    file.puts "    <o_orderdate> #{value(row[4], :date)} ;"
    file.puts "    <o_orderpriority> #{value(row[5], :text)} ;"
    file.puts "    <o_clerk> #{value(row[6], :text)} ;"
    file.puts "    <o_shippriority> #{value(row[7], :int)} ;"
    file.puts "    <o_comment> #{value(row[8], :text)} ."
  when 'lineitem'
    file.puts "<LineItem/#{row[0]}/#{row[3]}> a <LineItem> ;"
    file.puts "    <l_order> #{value(row[0], :order)} ;"
    file.puts "    <l_part> #{value(row[1], :part)} ;"
    file.puts "    <l_supp> #{value(row[2], :supplier)} ;"
    file.puts "    <l_linenumber> #{value(row[3], :int)} ;"
    file.puts "    <l_quantity> #{value(row[4], :decimal)} ;"
    file.puts "    <l_extendedprice> #{value(row[5], :decimal)} ;"
    file.puts "    <l_discount> #{value(row[6], :decimal)} ;"
    file.puts "    <l_tax> #{value(row[7], :decimal)} ;"
    file.puts "    <l_returnflag> #{value(row[8], :text)} ;"
    file.puts "    <l_linestatus> #{value(row[9], :text)} ;"
    file.puts "    <l_shipdate> #{value(row[10], :date)} ;"
    file.puts "    <l_commitdate> #{value(row[11], :date)} ;"
    file.puts "    <l_receiptdate> #{value(row[12], :date)} ;"
    file.puts "    <l_shipinstruct> #{value(row[13], :text)} ;"
    file.puts "    <l_shipmode> #{value(row[14], :text)} ;"
    file.puts "    <l_comment> #{value(row[15], :text)} ."
  end
end

[
  'part', 'region', 'nation', 'supplier',
  'partsupp', 'customer', 'orders', 'lineitem'
].each do |model|
  Dir[File.join(ARGV[0],"**","#{model}-sf*-br*.tbl.gz")].sort.each do |file|
    ofile = file.sub(/\.tbl\.gz$/, '.ttl.gz')
    puts "Proccessing #{file} -> #{ofile}"
    Zlib::GzipWriter.open(ofile) do |out_file|
      out_file.puts '@base <http://example.org/> .'
      out_file.puts '@prefix xsd: <http://www.w3.org/2001/XMLSchema#> .'
      Zlib::GzipReader.open(file) do |in_file|
        in_file.each_line do |line|
          parse_row(line.split('|'), model, out_file)
        end
      end
    end
  end
end
